<?php

use Illuminate\Support\Facades\Route;

// Controllers
use App\Http\Controllers\Auth\AuthController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\EventController as AdminEventController;
use App\Http\Controllers\Admin\RoomController;
use App\Http\Controllers\Client\EventController as ClientEventController;
use App\Http\Controllers\Client\AccountController;
use App\Http\Controllers\TPV\TPVController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
*/

// Health check
Route::get('/health', function () {
    return response()->json(['status' => 'ok', 'timestamp' => now()]);
});

// ==================== AUTH ROUTES ====================
Route::prefix('auth')->group(function () {
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);

    Route::middleware('jwt.auth')->group(function () {
        Route::post('/logout', [AuthController::class, 'logout']);
        Route::post('/refresh', [AuthController::class, 'refresh']);
        Route::get('/me', [AuthController::class, 'me']);
        Route::put('/profile', [AuthController::class, 'updateProfile']);
    });
});

// ==================== PUBLIC CLIENT ROUTES ====================
Route::prefix('events')->group(function () {
    Route::get('/', [ClientEventController::class, 'index']);
    Route::get('/{id}', [ClientEventController::class, 'show']);
    Route::get('/{id}/rooms', [ClientEventController::class, 'rooms']);
    Route::get('/{eventId}/rooms/{roomId}/tables', [ClientEventController::class, 'tables']);
});

// Order routes (can be guest or authenticated)
Route::post('/orders', [ClientEventController::class, 'createOrder']);
Route::get('/orders/{orderNumber}/status', [ClientEventController::class, 'orderStatus']);

// ==================== CLIENT AUTHENTICATED ROUTES ====================
Route::middleware(['jwt.auth', 'role:client,admin'])->prefix('me')->group(function () {
    Route::get('/orders', [AccountController::class, 'orders']);
    Route::get('/tickets', [AccountController::class, 'tickets']);
    Route::get('/points', [AccountController::class, 'points']);
    Route::get('/reservations', [AccountController::class, 'reservations']);
});

// ==================== ADMIN ROUTES ====================
Route::middleware(['jwt.auth', 'role:admin'])->prefix('admin')->group(function () {
    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'index']);

    // Events
    Route::apiResource('events', AdminEventController::class);
    Route::get('/events/{id}/stats', [AdminEventController::class, 'stats']);

    // Rooms
    Route::apiResource('rooms', RoomController::class);

    // Tables
    Route::get('/rooms/{roomId}/tables', [RoomController::class, 'tables']);
    Route::post('/rooms/{roomId}/tables', [RoomController::class, 'storeTable']);
    Route::put('/rooms/{roomId}/tables/positions', [RoomController::class, 'updateTablePositions']);
    Route::put('/tables/{tableId}', [RoomController::class, 'updateTable']);
    Route::patch('/tables/{tableId}/status', [RoomController::class, 'updateTableStatus']);
    Route::delete('/tables/{tableId}', [RoomController::class, 'destroyTable']);

    // Clients
    Route::get('/clients', function () {
        return \App\Models\User::where('role', 'client')
            ->withCount('orders')
            ->latest()
            ->paginate(20);
    });
    Route::get('/clients/{id}', function ($id) {
        return \App\Models\User::where('role', 'client')
            ->with(['orders.event:id,name', 'pointsHistory'])
            ->findOrFail($id);
    });

    // Settings
    Route::get('/settings', function () {
        return response()->json([
            'success' => true,
            'data' => [
                'branding' => \App\Models\Setting::getByGroup('branding'),
                'business' => \App\Models\Setting::getByGroup('business'),
                'points' => \App\Models\Setting::getByGroup('points'),
            ],
        ]);
    });
    Route::put('/settings', function (\Illuminate\Http\Request $request) {
        foreach ($request->all() as $key => $value) {
            if (is_array($value)) {
                foreach ($value as $k => $v) {
                    \App\Models\Setting::set($k, $v, $key);
                }
            }
        }
        return response()->json(['success' => true, 'message' => 'Configuración guardada']);
    });

    // Audit logs
    Route::get('/logs', function (\Illuminate\Http\Request $request) {
        return \App\Models\AuditLog::with('user:id,name,email')
            ->when($request->action, fn($q, $a) => $q->where('action', $a))
            ->when($request->user_id, fn($q, $u) => $q->where('user_id', $u))
            ->latest()
            ->paginate(50);
    });

    // Users management
    Route::get('/users', function () {
        return \App\Models\User::whereIn('role', ['admin', 'cashier'])->get();
    });
    Route::post('/users', function (\Illuminate\Http\Request $request) {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8',
            'role' => 'required|in:admin,cashier',
        ]);

        return \App\Models\User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'role' => $request->role,
            'status' => 'active',
        ]);
    });
});

// ==================== TPV ROUTES ====================
Route::middleware(['jwt.auth', 'role:cashier,admin'])->prefix('tpv')->group(function () {
    // Session management
    Route::post('/session/open', [TPVController::class, 'openSession']);
    Route::get('/session/current', [TPVController::class, 'currentSession']);
    Route::post('/session/close', [TPVController::class, 'closeSession']);

    // Sales
    Route::get('/events', [TPVController::class, 'events']);
    Route::post('/orders', [TPVController::class, 'createSale']);
    Route::get('/orders/history', [TPVController::class, 'salesHistory']);

    // Validation
    Route::post('/validate', [TPVController::class, 'validateTicket']);
});

// ==================== PAYMENT WEBHOOKS ====================
Route::post('/webhooks/flow', function (\Illuminate\Http\Request $request) {
    $flowService = app(\App\Services\FlowService::class);
    $result = $flowService->processWebhook($request->all());

    if ($result['success'] && $result['status'] === 2) { // 2 = paid
        $payment = \App\Models\Payment::where('flow_token', $request->token)->first();
        if ($payment) {
            $payment->markAsCompleted($result['data']);
            $payment->order->markAsPaid();

            // Generate tickets
            $qrService = app(\App\Services\QRService::class);
            $order = $payment->order->load('items');
            $quantity = $order->items->sum('quantity');

            for ($i = 0; $i < $quantity; $i++) {
                $ticket = $order->tickets()->create([
                    'event_id' => $order->event_id,
                    'status' => 'valid',
                ]);
                $qrService->generateForTicket($ticket->qr_code);
            }

            // Award points
            if ($order->user_id) {
                $pointsPerCurrency = \App\Models\Setting::get('points_per_currency', 1);
                $divisor = \App\Models\Setting::get('currency_divisor', 1000);
                $points = floor($order->total / $divisor) * $pointsPerCurrency;

                if ($points > 0) {
                    $order->user->addPoints($points, $order->id, 'Compra de entradas');
                }
            }
        }
    }

    return response()->json(['status' => 'ok']);
});
