<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected string $apiUrl;
    protected string $apiKey;
    protected string $phone;

    public function __construct()
    {
        $this->apiUrl = config('services.whatsapp.api_url');
        $this->apiKey = config('services.whatsapp.api_key');
        $this->phone = config('services.whatsapp.phone');
    }

    /**
     * Send a WhatsApp message
     */
    public function sendMessage(string $to, string $message): array
    {
        if (empty($this->apiUrl) || empty($this->apiKey)) {
            Log::warning('WhatsApp not configured');
            return ['success' => false, 'error' => 'WhatsApp no configurado'];
        }

        // Normalize phone number (Chile format)
        $to = $this->normalizePhone($to);

        try {
            $response = Http::withHeaders([
                'Authorization' => "Bearer {$this->apiKey}",
                'Content-Type' => 'application/json',
            ])->post("{$this->apiUrl}/messages", [
                        'to' => $to,
                        'type' => 'text',
                        'text' => ['body' => $message],
                    ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'message_id' => $response->json()['messages'][0]['id'] ?? null,
                ];
            }

            Log::error('WhatsApp send failed', ['response' => $response->json()]);
            return [
                'success' => false,
                'error' => 'Error al enviar mensaje',
            ];
        } catch (\Exception $e) {
            Log::error('WhatsApp exception', ['message' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => 'Error de conexión con WhatsApp',
            ];
        }
    }

    /**
     * Send order confirmation via WhatsApp
     */
    public function sendOrderConfirmation(string $to, array $orderData): array
    {
        $message = $this->buildOrderConfirmationMessage($orderData);
        return $this->sendMessage($to, $message);
    }

    /**
     * Send ticket QR via WhatsApp
     */
    public function sendTicketWithQR(string $to, array $ticketData, string $qrImageUrl): array
    {
        // First send text message
        $message = $this->buildTicketMessage($ticketData);
        $result = $this->sendMessage($to, $message);

        if (!$result['success']) {
            return $result;
        }

        // Then send QR image
        return $this->sendImage($to, $qrImageUrl, 'Tu código QR de entrada');
    }

    /**
     * Send an image via WhatsApp
     */
    public function sendImage(string $to, string $imageUrl, string $caption = ''): array
    {
        $to = $this->normalizePhone($to);

        try {
            $response = Http::withHeaders([
                'Authorization' => "Bearer {$this->apiKey}",
                'Content-Type' => 'application/json',
            ])->post("{$this->apiUrl}/messages", [
                        'to' => $to,
                        'type' => 'image',
                        'image' => [
                            'link' => $imageUrl,
                            'caption' => $caption,
                        ],
                    ]);

            return [
                'success' => $response->successful(),
                'error' => $response->successful() ? null : 'Error al enviar imagen',
            ];
        } catch (\Exception $e) {
            Log::error('WhatsApp image exception', ['message' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => 'Error de conexión',
            ];
        }
    }

    /**
     * Normalize phone number to international format
     */
    protected function normalizePhone(string $phone): string
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);

        // Add Chile country code if not present
        if (strlen($phone) === 9 && str_starts_with($phone, '9')) {
            $phone = '56' . $phone;
        } elseif (strlen($phone) === 8) {
            $phone = '569' . $phone;
        }

        return $phone;
    }

    /**
     * Build order confirmation message
     */
    protected function buildOrderConfirmationMessage(array $data): string
    {
        return "🎉 *¡Compra Confirmada!*\n\n" .
            "📋 Orden: {$data['order_number']}\n" .
            "🎪 Evento: {$data['event_name']}\n" .
            "📅 Fecha: {$data['event_date']}\n" .
            "🎟️ Entradas: {$data['quantity']}\n" .
            "💰 Total: \${$data['total']}\n\n" .
            "Recibirás tu código QR en breve.\n\n" .
            "¡Gracias por tu compra!";
    }

    /**
     * Build ticket message
     */
    protected function buildTicketMessage(array $data): string
    {
        return "🎟️ *Tu Entrada*\n\n" .
            "🎪 {$data['event_name']}\n" .
            "📅 {$data['event_date']}\n" .
            "⏰ {$data['event_time']}\n" .
            "🔢 Código: {$data['qr_code']}\n\n" .
            "Presenta este QR en la entrada.";
    }
}
