<?php

namespace App\Services;

use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Storage;

class QRService
{
    protected int $size;
    protected int $margin;
    protected string $format;

    public function __construct()
    {
        $this->size = config('services.qr.size', 300);
        $this->margin = config('services.qr.margin', 1);
        $this->format = config('services.qr.format', 'png');
    }

    /**
     * Generate QR code for a ticket
     */
    public function generateForTicket(string $qrCode): string
    {
        $filename = "qrcodes/{$qrCode}.{$this->format}";

        $qrImage = QrCode::format($this->format)
            ->size($this->size)
            ->margin($this->margin)
            ->errorCorrection('H')
            ->generate($qrCode);

        Storage::disk('public')->put($filename, $qrImage);

        return Storage::disk('public')->url($filename);
    }

    /**
     * Generate QR code with logo
     */
    public function generateWithLogo(string $data, string $logoPath): string
    {
        $filename = 'qrcodes/' . md5($data) . ".{$this->format}";

        $qrImage = QrCode::format($this->format)
            ->size($this->size)
            ->margin($this->margin)
            ->errorCorrection('H')
            ->merge($logoPath, 0.3, true)
            ->generate($data);

        Storage::disk('public')->put($filename, $qrImage);

        return Storage::disk('public')->url($filename);
    }

    /**
     * Generate QR as base64 for inline display
     */
    public function generateBase64(string $data): string
    {
        $qrImage = QrCode::format('png')
            ->size($this->size)
            ->margin($this->margin)
            ->errorCorrection('H')
            ->generate($data);

        return 'data:image/png;base64,' . base64_encode($qrImage);
    }

    /**
     * Generate QR as SVG string
     */
    public function generateSvg(string $data): string
    {
        return QrCode::format('svg')
            ->size($this->size)
            ->margin($this->margin)
            ->errorCorrection('H')
            ->generate($data);
    }

    /**
     * Delete QR code file
     */
    public function delete(string $qrCode): bool
    {
        $filename = "qrcodes/{$qrCode}.{$this->format}";
        return Storage::disk('public')->delete($filename);
    }

    /**
     * Check if QR code file exists
     */
    public function exists(string $qrCode): bool
    {
        $filename = "qrcodes/{$qrCode}.{$this->format}";
        return Storage::disk('public')->exists($filename);
    }

    /**
     * Get QR code URL
     */
    public function getUrl(string $qrCode): ?string
    {
        if ($this->exists($qrCode)) {
            return Storage::disk('public')->url("qrcodes/{$qrCode}.{$this->format}");
        }
        return null;
    }
}
