<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class FlowService
{
    protected string $apiKey;
    protected string $secretKey;
    protected string $apiUrl;
    protected string $returnUrl;
    protected string $webhookUrl;

    public function __construct()
    {
        $this->apiKey = config('services.flow.api_key');
        $this->secretKey = config('services.flow.secret_key');
        $this->apiUrl = config('services.flow.api_url');
        $this->returnUrl = config('services.flow.return_url');
        $this->webhookUrl = config('services.flow.webhook_url');
    }

    /**
     * Create a payment order in Flow
     */
    public function createPayment(array $data): array
    {
        $params = [
            'apiKey' => $this->apiKey,
            'commerceOrder' => $data['order_number'],
            'subject' => $data['subject'] ?? 'Compra de entrada',
            'currency' => 'CLP',
            'amount' => (int) $data['amount'],
            'email' => $data['email'],
            'urlConfirmation' => $this->webhookUrl,
            'urlReturn' => $this->returnUrl . '?order=' . $data['order_number'],
        ];

        if (isset($data['optional'])) {
            $params['optional'] = json_encode($data['optional']);
        }

        $params['s'] = $this->sign($params);

        try {
            $response = Http::asForm()->post("{$this->apiUrl}/payment/create", $params);

            if ($response->successful()) {
                $result = $response->json();
                return [
                    'success' => true,
                    'url' => $result['url'] . '?token=' . $result['token'],
                    'token' => $result['token'],
                    'flow_order' => $result['flowOrder'] ?? null,
                ];
            }

            Log::error('Flow payment creation failed', [
                'response' => $response->json(),
                'order' => $data['order_number'],
            ]);

            return [
                'success' => false,
                'error' => $response->json()['message'] ?? 'Error al crear pago',
            ];
        } catch (\Exception $e) {
            Log::error('Flow payment exception', [
                'message' => $e->getMessage(),
                'order' => $data['order_number'],
            ]);

            return [
                'success' => false,
                'error' => 'Error de conexión con Flow',
            ];
        }
    }

    /**
     * Get payment status from Flow
     */
    public function getPaymentStatus(string $token): array
    {
        $params = [
            'apiKey' => $this->apiKey,
            'token' => $token,
        ];

        $params['s'] = $this->sign($params);

        try {
            $response = Http::asForm()->post("{$this->apiUrl}/payment/getStatus", $params);

            if ($response->successful()) {
                $result = $response->json();
                return [
                    'success' => true,
                    'status' => $result['status'], // 1=pending, 2=paid, 3=rejected, 4=cancelled
                    'data' => $result,
                ];
            }

            return [
                'success' => false,
                'error' => 'Error al obtener estado del pago',
            ];
        } catch (\Exception $e) {
            Log::error('Flow get status exception', ['message' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => 'Error de conexión con Flow',
            ];
        }
    }

    /**
     * Process webhook callback from Flow
     */
    public function processWebhook(array $data): array
    {
        if (!isset($data['token'])) {
            return ['success' => false, 'error' => 'Token no proporcionado'];
        }

        return $this->getPaymentStatus($data['token']);
    }

    /**
     * Generate HMAC signature for Flow API
     */
    protected function sign(array $params): string
    {
        ksort($params);
        $toSign = '';
        foreach ($params as $key => $value) {
            $toSign .= $key . $value;
        }
        return hash_hmac('sha256', $toSign, $this->secretKey);
    }

    /**
     * Refund a payment (if supported)
     */
    public function refund(string $token, float $amount): array
    {
        $params = [
            'apiKey' => $this->apiKey,
            'token' => $token,
            'amount' => (int) $amount,
        ];

        $params['s'] = $this->sign($params);

        try {
            $response = Http::asForm()->post("{$this->apiUrl}/payment/refund", $params);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                ];
            }

            return [
                'success' => false,
                'error' => $response->json()['message'] ?? 'Error al procesar reembolso',
            ];
        } catch (\Exception $e) {
            Log::error('Flow refund exception', ['message' => $e->getMessage()]);
            return [
                'success' => false,
                'error' => 'Error de conexión con Flow',
            ];
        }
    }
}
