<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Tymon\JWTAuth\Contracts\JWTSubject;

class User extends Authenticatable implements JWTSubject
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'password',
        'role',
        'points',
        'status',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'points' => 'integer',
    ];

    // JWT Methods
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    public function getJWTCustomClaims()
    {
        return [
            'role' => $this->role,
            'name' => $this->name,
        ];
    }

    // Relationships
    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function reservations()
    {
        return $this->hasMany(Reservation::class);
    }

    public function pointsHistory()
    {
        return $this->hasMany(UserPoint::class);
    }

    public function cashierSessions()
    {
        return $this->hasMany(CashierSession::class);
    }

    public function auditLogs()
    {
        return $this->hasMany(AuditLog::class);
    }

    // Helpers
    public function isAdmin(): bool
    {
        return $this->role === 'admin';
    }

    public function isCashier(): bool
    {
        return $this->role === 'cashier';
    }

    public function isClient(): bool
    {
        return $this->role === 'client';
    }

    public function isActive(): bool
    {
        return $this->status === 'active';
    }

    public function addPoints(int $points, ?int $orderId = null, ?string $description = null): void
    {
        $this->pointsHistory()->create([
            'points' => $points,
            'type' => 'earned',
            'order_id' => $orderId,
            'description' => $description,
        ]);

        $this->increment('points', $points);
    }

    public function redeemPoints(int $points, ?string $description = null): bool
    {
        if ($this->points < $points) {
            return false;
        }

        $this->pointsHistory()->create([
            'points' => -$points,
            'type' => 'redeemed',
            'description' => $description,
        ]);

        $this->decrement('points', $points);
        return true;
    }
}
