<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Ticket extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'event_id',
        'qr_code',
        'status',
        'used_at',
        'validated_by',
    ];

    protected $casts = [
        'used_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($ticket) {
            if (empty($ticket->qr_code)) {
                $ticket->qr_code = self::generateQRCode();
            }
        });
    }

    public static function generateQRCode(): string
    {
        return 'TKT-' . strtoupper(Str::random(12));
    }

    // Relationships
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public function event()
    {
        return $this->belongsTo(Event::class);
    }

    public function validator()
    {
        return $this->belongsTo(User::class, 'validated_by');
    }

    // Scopes
    public function scopeValid($query)
    {
        return $query->where('status', 'valid');
    }

    public function scopeUsed($query)
    {
        return $query->where('status', 'used');
    }

    // Helpers
    public function isValid(): bool
    {
        return $this->status === 'valid';
    }

    public function isUsed(): bool
    {
        return $this->status === 'used';
    }

    public function isCancelled(): bool
    {
        return $this->status === 'cancelled';
    }

    public function markAsUsed(int $validatedBy): void
    {
        $this->update([
            'status' => 'used',
            'used_at' => now(),
            'validated_by' => $validatedBy,
        ]);
    }

    public function cancel(): void
    {
        $this->update(['status' => 'cancelled']);
    }

    public function getQRImagePath(): string
    {
        return storage_path("app/public/qrcodes/{$this->qr_code}.png");
    }
}
