<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Table extends Model
{
    use HasFactory;

    protected $fillable = [
        'room_id',
        'name',
        'capacity',
        'position_x',
        'position_y',
        'width',
        'height',
        'shape',
        'status',
    ];

    protected $casts = [
        'capacity' => 'integer',
        'position_x' => 'integer',
        'position_y' => 'integer',
        'width' => 'integer',
        'height' => 'integer',
    ];

    // Relationships
    public function room()
    {
        return $this->belongsTo(Room::class);
    }

    public function reservations()
    {
        return $this->hasMany(Reservation::class);
    }

    // Scopes
    public function scopeAvailable($query)
    {
        return $query->where('status', 'available');
    }

    public function scopeReserved($query)
    {
        return $query->where('status', 'reserved');
    }

    // Helpers
    public function isAvailable(): bool
    {
        return $this->status === 'available';
    }

    public function isReserved(): bool
    {
        return $this->status === 'reserved';
    }

    public function isBlocked(): bool
    {
        return $this->status === 'blocked';
    }

    public function markAsReserved(): void
    {
        $this->update(['status' => 'reserved']);
    }

    public function markAsAvailable(): void
    {
        $this->update(['status' => 'available']);
    }

    public function markAsBlocked(): void
    {
        $this->update(['status' => 'blocked']);
    }

    public function isAvailableForEvent(int $eventId): bool
    {
        return !$this->reservations()
            ->where('event_id', $eventId)
            ->whereIn('status', ['pending', 'confirmed'])
            ->exists();
    }
}
