<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'group',
        'type',
    ];

    public static function get(string $key, $default = null)
    {
        $setting = self::where('key', $key)->first();

        if (!$setting) {
            return $default;
        }

        return self::castValue($setting->value, $setting->type);
    }

    public static function set(string $key, $value, string $group = 'general', string $type = 'string'): self
    {
        return self::updateOrCreate(
            ['key' => $key],
            [
                'value' => is_array($value) ? json_encode($value) : $value,
                'group' => $group,
                'type' => $type,
            ]
        );
    }

    public static function getByGroup(string $group): array
    {
        return self::where('group', $group)
            ->get()
            ->mapWithKeys(function ($setting) {
                return [$setting->key => self::castValue($setting->value, $setting->type)];
            })
            ->toArray();
    }

    protected static function castValue($value, string $type)
    {
        return match ($type) {
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'number' => is_numeric($value) ? (float) $value : 0,
            'json' => json_decode($value, true),
            default => $value,
        };
    }

    // Default settings
    public static function defaults(): array
    {
        return [
            // Branding
            'app_name' => ['value' => 'Sistema Reservas', 'group' => 'branding', 'type' => 'string'],
            'app_logo' => ['value' => '', 'group' => 'branding', 'type' => 'string'],
            'primary_color' => ['value' => '#6366f1', 'group' => 'branding', 'type' => 'string'],
            'secondary_color' => ['value' => '#ec4899', 'group' => 'branding', 'type' => 'string'],

            // Business
            'business_name' => ['value' => '', 'group' => 'business', 'type' => 'string'],
            'business_rut' => ['value' => '', 'group' => 'business', 'type' => 'string'],
            'business_address' => ['value' => '', 'group' => 'business', 'type' => 'string'],
            'business_phone' => ['value' => '', 'group' => 'business', 'type' => 'string'],
            'business_email' => ['value' => '', 'group' => 'business', 'type' => 'string'],

            // Points
            'points_enabled' => ['value' => 'true', 'group' => 'points', 'type' => 'boolean'],
            'points_per_currency' => ['value' => '1', 'group' => 'points', 'type' => 'number'],
            'currency_divisor' => ['value' => '1000', 'group' => 'points', 'type' => 'number'],
        ];
    }
}
