<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Event extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'image',
        'start_date',
        'end_date',
        'start_time',
        'end_time',
        'type',
        'ticket_price',
        'total_capacity',
        'status',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'start_time' => 'datetime:H:i',
        'end_time' => 'datetime:H:i',
        'ticket_price' => 'decimal:2',
        'total_capacity' => 'integer',
    ];

    protected $appends = [
        'tickets_sold',
        'tickets_available',
        'total_revenue',
    ];

    // Relationships
    public function rooms()
    {
        return $this->belongsToMany(Room::class, 'event_rooms')
            ->withPivot('capacity_override')
            ->withTimestamps();
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function tickets()
    {
        return $this->hasMany(Ticket::class);
    }

    public function reservations()
    {
        return $this->hasMany(Reservation::class);
    }

    // Accessors
    public function getTicketsSoldAttribute(): int
    {
        return $this->tickets()
            ->whereIn('status', ['valid', 'used'])
            ->count();
    }

    public function getTicketsAvailableAttribute(): int
    {
        return max(0, $this->total_capacity - $this->tickets_sold);
    }

    public function getTotalRevenueAttribute(): float
    {
        return $this->orders()
            ->where('status', 'paid')
            ->sum('total');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeUpcoming($query)
    {
        return $query->where('start_date', '>=', now()->toDateString());
    }

    public function scopePublic($query)
    {
        return $query->whereIn('status', ['active']);
    }

    // Helpers
    public function isFree(): bool
    {
        return $this->type === 'free';
    }

    public function isPaid(): bool
    {
        return $this->type === 'paid';
    }

    public function hasAvailableTickets(int $quantity = 1): bool
    {
        return $this->tickets_available >= $quantity;
    }
}
