<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CashierSession extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'opening_amount',
        'closing_amount',
        'expected_amount',
        'opened_at',
        'closed_at',
        'notes',
    ];

    protected $casts = [
        'opening_amount' => 'decimal:2',
        'closing_amount' => 'decimal:2',
        'expected_amount' => 'decimal:2',
        'opened_at' => 'datetime',
        'closed_at' => 'datetime',
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    // Scopes
    public function scopeOpen($query)
    {
        return $query->whereNull('closed_at');
    }

    public function scopeClosed($query)
    {
        return $query->whereNotNull('closed_at');
    }

    // Helpers
    public function isOpen(): bool
    {
        return is_null($this->closed_at);
    }

    public function isClosed(): bool
    {
        return !is_null($this->closed_at);
    }

    public function getTotalSales(): float
    {
        return $this->orders()
            ->where('status', 'paid')
            ->sum('total');
    }

    public function getCashSales(): float
    {
        return $this->orders()
            ->where('status', 'paid')
            ->whereHas('payments', function ($query) {
                $query->where('method', 'cash')->where('status', 'completed');
            })
            ->sum('total');
    }

    public function close(float $closingAmount, ?string $notes = null): void
    {
        $expectedAmount = $this->opening_amount + $this->getCashSales();

        $this->update([
            'closing_amount' => $closingAmount,
            'expected_amount' => $expectedAmount,
            'closed_at' => now(),
            'notes' => $notes,
        ]);
    }

    public function getDifference(): float
    {
        if (!$this->isClosed()) {
            return 0;
        }
        return $this->closing_amount - $this->expected_amount;
    }
}
