<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AuditLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'action',
        'model',
        'model_id',
        'old_values',
        'new_values',
        'ip_address',
        'user_agent',
    ];

    protected $casts = [
        'old_values' => 'array',
        'new_values' => 'array',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Static helpers for logging
    public static function log(
        string $action,
        ?string $model = null,
        ?int $modelId = null,
        ?array $oldValues = null,
        ?array $newValues = null
    ): self {
        return self::create([
            'user_id' => auth()->id(),
            'action' => $action,
            'model' => $model,
            'model_id' => $modelId,
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }

    public static function logLogin(): self
    {
        return self::log('login');
    }

    public static function logLogout(): self
    {
        return self::log('logout');
    }

    public static function logCreate(Model $model): self
    {
        return self::log(
            'create',
            get_class($model),
            $model->id,
            null,
            $model->toArray()
        );
    }

    public static function logUpdate(Model $model, array $oldValues): self
    {
        return self::log(
            'update',
            get_class($model),
            $model->id,
            $oldValues,
            $model->toArray()
        );
    }

    public static function logDelete(Model $model): self
    {
        return self::log(
            'delete',
            get_class($model),
            $model->id,
            $model->toArray(),
            null
        );
    }
}
