<?php

namespace App\Http\Controllers\TPV;

use App\Http\Controllers\Controller;
use App\Models\CashierSession;
use App\Models\Event;
use App\Models\Order;
use App\Models\Ticket;
use App\Services\FlowService;
use App\Services\QRService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class TPVController extends Controller
{
    protected FlowService $flowService;
    protected QRService $qrService;

    public function __construct(FlowService $flowService, QRService $qrService)
    {
        $this->flowService = $flowService;
        $this->qrService = $qrService;
    }

    // ==================== SESSION MANAGEMENT ====================

    /**
     * Open cashier session
     */
    public function openSession(Request $request): JsonResponse
    {
        $user = auth()->user();

        // Check if user already has an open session
        $existingSession = CashierSession::where('user_id', $user->id)
            ->open()
            ->first();

        if ($existingSession) {
            return $this->error('Ya tienes una sesión de caja abierta', 400);
        }

        $validator = Validator::make($request->all(), [
            'opening_amount' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $session = CashierSession::create([
            'user_id' => $user->id,
            'opening_amount' => $request->opening_amount,
            'opened_at' => now(),
        ]);

        return $this->success($session, 'Sesión de caja abierta');
    }

    /**
     * Get current session
     */
    public function currentSession(): JsonResponse
    {
        $session = CashierSession::where('user_id', auth()->id())
            ->open()
            ->first();

        if (!$session) {
            return $this->error('No hay sesión de caja abierta', 404);
        }

        return $this->success([
            'session' => $session,
            'sales_summary' => [
                'total_sales' => $session->getTotalSales(),
                'cash_sales' => $session->getCashSales(),
                'orders_count' => $session->orders()->where('status', 'paid')->count(),
            ],
        ]);
    }

    /**
     * Close cashier session
     */
    public function closeSession(Request $request): JsonResponse
    {
        $session = CashierSession::where('user_id', auth()->id())
            ->open()
            ->first();

        if (!$session) {
            return $this->error('No hay sesión de caja abierta', 404);
        }

        $validator = Validator::make($request->all(), [
            'closing_amount' => 'required|numeric|min:0',
            'notes' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $session->close($request->closing_amount, $request->notes);

        return $this->success([
            'session' => $session->fresh(),
            'difference' => $session->getDifference(),
        ], 'Sesión de caja cerrada');
    }

    // ==================== SALES ====================

    /**
     * Get active events for TPV
     */
    public function events(): JsonResponse
    {
        $events = Event::active()
            ->orderBy('start_date')
            ->get()
            ->map(function ($event) {
                return [
                    'id' => $event->id,
                    'name' => $event->name,
                    'date' => $event->start_date->format('Y-m-d'),
                    'time' => $event->start_time?->format('H:i'),
                    'price' => $event->ticket_price,
                    'available' => $event->tickets_available,
                    'type' => $event->type,
                ];
            });

        return $this->success($events);
    }

    /**
     * Create TPV sale
     */
    public function createSale(Request $request): JsonResponse
    {
        // Check for open session
        $session = CashierSession::where('user_id', auth()->id())
            ->open()
            ->first();

        if (!$session) {
            return $this->error('Debes abrir una sesión de caja primero', 400);
        }

        $validator = Validator::make($request->all(), [
            'event_id' => 'required|exists:events,id',
            'quantity' => 'required|integer|min:1|max:20',
            'payment_method' => 'required|in:cash,credit,debit',
            'customer_email' => 'nullable|email',
            'customer_name' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $event = Event::findOrFail($request->event_id);

        if (!$event->hasAvailableTickets($request->quantity)) {
            return $this->error('No hay suficientes entradas disponibles', 400);
        }

        DB::beginTransaction();
        try {
            // Create order
            $order = Order::create([
                'event_id' => $event->id,
                'cashier_session_id' => $session->id,
                'subtotal' => $event->ticket_price * $request->quantity,
                'total' => $event->ticket_price * $request->quantity,
                'status' => 'pending',
                'channel' => 'tpv',
            ]);

            $order->addItem(
                'ticket',
                $request->quantity,
                $event->ticket_price,
                "Entrada - {$event->name}"
            );

            // Handle payment
            if ($request->payment_method === 'cash' || $event->isFree()) {
                // Direct cash payment
                $order->payments()->create([
                    'method' => 'cash',
                    'amount' => $order->total,
                    'status' => 'completed',
                    'paid_at' => now(),
                ]);

                $order->markAsPaid();

                // Generate tickets
                $tickets = $this->generateTickets($order, $request->quantity);

                DB::commit();

                return $this->success([
                    'order' => $order->fresh(),
                    'tickets' => $tickets,
                    'print_data' => $this->generatePrintData($order, $tickets),
                ], 'Venta completada');
            } else {
                // Card payment via Flow
                $paymentResult = $this->flowService->createPayment([
                    'order_number' => $order->order_number,
                    'amount' => $order->total,
                    'email' => $request->customer_email ?? 'tpv@sistema.local',
                    'subject' => "Entrada - {$event->name}",
                ]);

                if (!$paymentResult['success']) {
                    DB::rollBack();
                    return $this->error('Error al procesar pago: ' . $paymentResult['error'], 400);
                }

                $order->payments()->create([
                    'flow_token' => $paymentResult['token'],
                    'method' => $request->payment_method,
                    'amount' => $order->total,
                    'status' => 'pending',
                ]);

                DB::commit();

                return $this->success([
                    'order_number' => $order->order_number,
                    'payment_url' => $paymentResult['url'],
                    'awaiting_payment' => true,
                ], 'Esperando pago con tarjeta');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->error('Error en la venta: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Generate tickets for order
     */
    protected function generateTickets(Order $order, int $quantity): array
    {
        $tickets = [];
        for ($i = 0; $i < $quantity; $i++) {
            $ticket = $order->tickets()->create([
                'event_id' => $order->event_id,
                'status' => 'valid',
            ]);

            $this->qrService->generateForTicket($ticket->qr_code);

            $tickets[] = [
                'id' => $ticket->id,
                'qr_code' => $ticket->qr_code,
                'qr_url' => $this->qrService->getUrl($ticket->qr_code),
            ];
        }
        return $tickets;
    }

    /**
     * Generate print data for receipt
     */
    protected function generatePrintData(Order $order, array $tickets): array
    {
        $order->load('event');

        return [
            'business_name' => \App\Models\Setting::get('business_name', 'Sistema Reservas'),
            'business_rut' => \App\Models\Setting::get('business_rut', ''),
            'business_address' => \App\Models\Setting::get('business_address', ''),
            'order_number' => $order->order_number,
            'event_name' => $order->event->name,
            'event_date' => $order->event->start_date->format('d/m/Y'),
            'event_time' => $order->event->start_time?->format('H:i'),
            'quantity' => count($tickets),
            'unit_price' => $order->event->ticket_price,
            'total' => $order->total,
            'date' => now()->format('d/m/Y H:i'),
            'tickets' => $tickets,
        ];
    }

    // ==================== VALIDATION ====================

    /**
     * Validate ticket QR
     */
    public function validateTicket(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'qr_code' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->error('Código QR requerido', 422);
        }

        $ticket = Ticket::where('qr_code', $request->qr_code)
            ->with(['event:id,name,start_date,start_time', 'order:id,order_number'])
            ->first();

        if (!$ticket) {
            return $this->success([
                'valid' => false,
                'status' => 'not_found',
                'message' => 'Entrada no encontrada',
            ]);
        }

        if ($ticket->isUsed()) {
            return $this->success([
                'valid' => false,
                'status' => 'used',
                'message' => 'Entrada ya utilizada',
                'used_at' => $ticket->used_at->format('d/m/Y H:i'),
                'event' => $ticket->event->name,
            ]);
        }

        if ($ticket->isCancelled()) {
            return $this->success([
                'valid' => false,
                'status' => 'cancelled',
                'message' => 'Entrada cancelada',
            ]);
        }

        // Mark as used
        $ticket->markAsUsed(auth()->id());

        return $this->success([
            'valid' => true,
            'status' => 'valid',
            'message' => '¡Entrada válida!',
            'event' => $ticket->event->name,
            'event_date' => $ticket->event->start_date->format('d/m/Y'),
            'order_number' => $ticket->order->order_number,
        ]);
    }

    /**
     * Get TPV sales history
     */
    public function salesHistory(Request $request): JsonResponse
    {
        $session = CashierSession::where('user_id', auth()->id())
            ->open()
            ->first();

        if (!$session) {
            return $this->error('No hay sesión activa', 404);
        }

        $orders = $session->orders()
            ->with(['event:id,name', 'payments'])
            ->latest()
            ->paginate(20);

        return $this->paginated($orders);
    }
}
