<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\Room;
use App\Models\Table;
use App\Models\Order;
use App\Models\Ticket;
use App\Services\FlowService;
use App\Services\QRService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class EventController extends Controller
{
    protected FlowService $flowService;
    protected QRService $qrService;

    public function __construct(FlowService $flowService, QRService $qrService)
    {
        $this->flowService = $flowService;
        $this->qrService = $qrService;
    }

    /**
     * List active events for clients
     */
    public function index(Request $request): JsonResponse
    {
        $query = Event::active()->upcoming();

        if ($request->has('date')) {
            $query->whereDate('start_date', $request->date);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        $events = $query->orderBy('start_date')
            ->select([
                'id',
                'name',
                'description',
                'image',
                'start_date',
                'end_date',
                'start_time',
                'end_time',
                'type',
                'ticket_price',
                'total_capacity',
                'status'
            ])
            ->get()
            ->map(function ($event) {
                return [
                    'id' => $event->id,
                    'name' => $event->name,
                    'description' => $event->description,
                    'image' => $event->image ? asset('storage/' . $event->image) : null,
                    'start_date' => $event->start_date->format('Y-m-d'),
                    'end_date' => $event->end_date?->format('Y-m-d'),
                    'start_time' => $event->start_time?->format('H:i'),
                    'end_time' => $event->end_time?->format('H:i'),
                    'type' => $event->type,
                    'ticket_price' => $event->ticket_price,
                    'tickets_available' => $event->tickets_available,
                    'is_sold_out' => $event->tickets_available <= 0,
                ];
            });

        return $this->success($events);
    }

    /**
     * Get event details
     */
    public function show(int $id): JsonResponse
    {
        $event = Event::active()->findOrFail($id);

        return $this->success([
            'id' => $event->id,
            'name' => $event->name,
            'description' => $event->description,
            'image' => $event->image ? asset('storage/' . $event->image) : null,
            'start_date' => $event->start_date->format('Y-m-d'),
            'end_date' => $event->end_date?->format('Y-m-d'),
            'start_time' => $event->start_time?->format('H:i'),
            'end_time' => $event->end_time?->format('H:i'),
            'type' => $event->type,
            'ticket_price' => $event->ticket_price,
            'tickets_available' => $event->tickets_available,
            'is_sold_out' => $event->tickets_available <= 0,
        ]);
    }

    /**
     * Get rooms for an event
     */
    public function rooms(int $eventId): JsonResponse
    {
        $event = Event::active()->findOrFail($eventId);

        $rooms = $event->rooms()
            ->where('is_active', true)
            ->get()
            ->map(function ($room) use ($eventId) {
                return [
                    'id' => $room->id,
                    'name' => $room->name,
                    'capacity' => $room->pivot->capacity_override ?? $room->capacity,
                    'available_tables' => $room->tables()
                        ->where('status', 'available')
                        ->count(),
                ];
            });

        return $this->success($rooms);
    }

    /**
     * Get available tables for event/room
     */
    public function tables(int $eventId, int $roomId): JsonResponse
    {
        Event::active()->findOrFail($eventId);
        $room = Room::findOrFail($roomId);

        $tables = $room->tables()
            ->get()
            ->map(function ($table) use ($eventId) {
                $isAvailable = $table->isAvailableForEvent($eventId) && $table->status !== 'blocked';
                return [
                    'id' => $table->id,
                    'name' => $table->name,
                    'capacity' => $table->capacity,
                    'position_x' => $table->position_x,
                    'position_y' => $table->position_y,
                    'width' => $table->width,
                    'height' => $table->height,
                    'shape' => $table->shape,
                    'is_available' => $isAvailable,
                    'status' => $isAvailable ? 'available' : 'reserved',
                ];
            });

        return $this->success($tables);
    }

    /**
     * Create order for tickets
     */
    public function createOrder(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'event_id' => 'required|exists:events,id',
            'quantity' => 'required|integer|min:1|max:10',
            'table_id' => 'nullable|exists:tables,id',
            'email' => 'required_without:user|email',
            'name' => 'required_without:user|string|max:255',
            'phone' => 'nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $event = Event::active()->findOrFail($request->event_id);

        // Check availability
        if (!$event->hasAvailableTickets($request->quantity)) {
            return $this->error('No hay suficientes entradas disponibles', 400);
        }

        // Check table availability
        if ($request->table_id) {
            $table = Table::findOrFail($request->table_id);
            if (!$table->isAvailableForEvent($event->id)) {
                return $this->error('La mesa seleccionada no está disponible', 400);
            }
        }

        DB::beginTransaction();
        try {
            // Create or get user
            $user = auth()->user();
            if (!$user && $request->email) {
                $user = \App\Models\User::firstOrCreate(
                    ['email' => $request->email],
                    [
                        'name' => $request->name,
                        'phone' => $request->phone,
                        'password' => bcrypt(\Illuminate\Support\Str::random(16)),
                        'role' => 'client',
                    ]
                );
            }

            // Create order
            $order = Order::create([
                'user_id' => $user?->id,
                'event_id' => $event->id,
                'subtotal' => $event->ticket_price * $request->quantity,
                'total' => $event->ticket_price * $request->quantity,
                'status' => 'pending',
                'channel' => 'web',
            ]);

            // Add order item
            $order->addItem(
                'ticket',
                $request->quantity,
                $event->ticket_price,
                "Entrada - {$event->name}"
            );

            // Create table reservation if selected
            if ($request->table_id) {
                \App\Models\Reservation::create([
                    'user_id' => $user->id,
                    'table_id' => $request->table_id,
                    'event_id' => $event->id,
                    'reserved_at' => $event->start_date->setTimeFromTimeString($event->start_time->format('H:i:s')),
                    'guests' => $request->quantity,
                    'status' => 'pending',
                ]);
            }

            DB::commit();

            // If free event, mark as paid immediately
            if ($event->isFree()) {
                $order->markAsPaid();
                $this->generateTickets($order, $request->quantity);

                return $this->success([
                    'order' => $order->fresh(),
                    'message' => 'Entrada confirmada',
                ], 'Orden creada');
            }

            // Initiate payment with Flow
            $paymentResult = $this->flowService->createPayment([
                'order_number' => $order->order_number,
                'amount' => $order->total,
                'email' => $user?->email ?? $request->email,
                'subject' => "Entrada - {$event->name}",
            ]);

            if (!$paymentResult['success']) {
                DB::rollBack();
                return $this->error($paymentResult['error'], 400);
            }

            // Save payment record
            $order->payments()->create([
                'flow_token' => $paymentResult['token'],
                'method' => 'credit',
                'amount' => $order->total,
                'status' => 'pending',
            ]);

            return $this->success([
                'order_number' => $order->order_number,
                'payment_url' => $paymentResult['url'],
            ], 'Redirigiendo al pago');

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->error('Error al crear la orden: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Generate tickets for paid order
     */
    protected function generateTickets(Order $order, int $quantity): void
    {
        for ($i = 0; $i < $quantity; $i++) {
            $ticket = $order->tickets()->create([
                'event_id' => $order->event_id,
                'status' => 'valid',
            ]);

            // Generate QR code
            $this->qrService->generateForTicket($ticket->qr_code);
        }
    }

    /**
     * Get order status
     */
    public function orderStatus(string $orderNumber): JsonResponse
    {
        $order = Order::where('order_number', $orderNumber)
            ->with(['event:id,name,start_date,start_time', 'tickets'])
            ->firstOrFail();

        return $this->success([
            'order_number' => $order->order_number,
            'status' => $order->status,
            'total' => $order->total,
            'event' => $order->event,
            'tickets' => $order->tickets->map(fn($t) => [
                'qr_code' => $t->qr_code,
                'qr_url' => $this->qrService->getUrl($t->qr_code),
                'status' => $t->status,
            ]),
        ]);
    }
}
