<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Ticket;
use Illuminate\Http\JsonResponse;

class AccountController extends Controller
{
    /**
     * Get user's orders
     */
    public function orders(): JsonResponse
    {
        $orders = auth()->user()
            ->orders()
            ->with(['event:id,name,start_date,start_time,image'])
            ->latest()
            ->paginate(10);

        return $this->paginated($orders, function ($order) {
            return [
                'id' => $order->id,
                'order_number' => $order->order_number,
                'event' => [
                    'id' => $order->event->id,
                    'name' => $order->event->name,
                    'date' => $order->event->start_date->format('Y-m-d'),
                    'time' => $order->event->start_time?->format('H:i'),
                    'image' => $order->event->image ? asset('storage/' . $order->event->image) : null,
                ],
                'total' => $order->total,
                'status' => $order->status,
                'created_at' => $order->created_at->toISOString(),
            ];
        });
    }

    /**
     * Get user's active tickets
     */
    public function tickets(): JsonResponse
    {
        $tickets = Ticket::whereHas('order', function ($query) {
            $query->where('user_id', auth()->id());
        })
            ->with(['event:id,name,start_date,start_time'])
            ->where('status', 'valid')
            ->get()
            ->map(function ($ticket) {
                return [
                    'id' => $ticket->id,
                    'qr_code' => $ticket->qr_code,
                    'qr_url' => asset('storage/qrcodes/' . $ticket->qr_code . '.png'),
                    'event' => [
                        'id' => $ticket->event->id,
                        'name' => $ticket->event->name,
                        'date' => $ticket->event->start_date->format('Y-m-d'),
                        'time' => $ticket->event->start_time?->format('H:i'),
                    ],
                    'status' => $ticket->status,
                ];
            });

        return $this->success($tickets);
    }

    /**
     * Get user's points history
     */
    public function points(): JsonResponse
    {
        $user = auth()->user();

        $history = $user->pointsHistory()
            ->with('order:id,order_number')
            ->latest()
            ->limit(20)
            ->get()
            ->map(function ($point) {
                return [
                    'id' => $point->id,
                    'points' => $point->points,
                    'type' => $point->type,
                    'description' => $point->description,
                    'order_number' => $point->order?->order_number,
                    'created_at' => $point->created_at->toISOString(),
                ];
            });

        return $this->success([
            'total_points' => $user->points,
            'history' => $history,
        ]);
    }

    /**
     * Get user's reservations
     */
    public function reservations(): JsonResponse
    {
        $reservations = auth()->user()
            ->reservations()
            ->with([
                'event:id,name,start_date,start_time',
                'table:id,name,capacity',
                'table.room:id,name',
            ])
            ->whereIn('status', ['pending', 'confirmed'])
            ->orderBy('reserved_at')
            ->get()
            ->map(function ($reservation) {
                return [
                    'id' => $reservation->id,
                    'event' => [
                        'id' => $reservation->event->id,
                        'name' => $reservation->event->name,
                        'date' => $reservation->event->start_date->format('Y-m-d'),
                        'time' => $reservation->event->start_time?->format('H:i'),
                    ],
                    'table' => [
                        'id' => $reservation->table->id,
                        'name' => $reservation->table->name,
                        'capacity' => $reservation->table->capacity,
                        'room' => $reservation->table->room->name,
                    ],
                    'guests' => $reservation->guests,
                    'status' => $reservation->status,
                    'reserved_at' => $reservation->reserved_at->toISOString(),
                ];
            });

        return $this->success($reservations);
    }
}
