<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Room;
use App\Models\Table;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class RoomController extends Controller
{
    /**
     * List all rooms
     */
    public function index(): JsonResponse
    {
        $rooms = Room::withCount('tables')
            ->with([
                'tables' => function ($query) {
                    $query->select('id', 'room_id', 'name', 'capacity', 'status', 'position_x', 'position_y', 'width', 'height', 'shape');
                }
            ])
            ->orderBy('name')
            ->get();

        return $this->success($rooms);
    }

    /**
     * Create a new room
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'capacity' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $room = Room::create($request->all());
        AuditLog::logCreate($room);

        return $this->success($room, 'Salón creado', 201);
    }

    /**
     * Get room details
     */
    public function show(int $id): JsonResponse
    {
        $room = Room::with('tables')->findOrFail($id);
        return $this->success($room);
    }

    /**
     * Update a room
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $room = Room::findOrFail($id);
        $oldValues = $room->toArray();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'capacity' => 'sometimes|integer|min:1',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $room->update($request->all());
        AuditLog::logUpdate($room, $oldValues);

        return $this->success($room, 'Salón actualizado');
    }

    /**
     * Delete a room
     */
    public function destroy(int $id): JsonResponse
    {
        $room = Room::findOrFail($id);

        if ($room->tables()->exists()) {
            return $this->error('No se puede eliminar un salón con mesas', 400);
        }

        AuditLog::logDelete($room);
        $room->delete();

        return $this->success(null, 'Salón eliminado');
    }

    // ==================== TABLES ====================

    /**
     * List tables in a room
     */
    public function tables(int $roomId): JsonResponse
    {
        $room = Room::findOrFail($roomId);
        return $this->success($room->tables);
    }

    /**
     * Create a table in a room
     */
    public function storeTable(Request $request, int $roomId): JsonResponse
    {
        $room = Room::findOrFail($roomId);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'capacity' => 'required|integer|min:1|max:20',
            'position_x' => 'required|integer|min:0',
            'position_y' => 'required|integer|min:0',
            'width' => 'integer|min:40|max:200',
            'height' => 'integer|min:40|max:200',
            'shape' => 'in:circle,square,rectangle',
            'status' => 'in:available,blocked',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $table = $room->tables()->create($request->all());
        AuditLog::logCreate($table);

        return $this->success($table, 'Mesa creada', 201);
    }

    /**
     * Update a table
     */
    public function updateTable(Request $request, int $tableId): JsonResponse
    {
        $table = Table::findOrFail($tableId);
        $oldValues = $table->toArray();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'capacity' => 'sometimes|integer|min:1|max:20',
            'position_x' => 'sometimes|integer|min:0',
            'position_y' => 'sometimes|integer|min:0',
            'width' => 'integer|min:40|max:200',
            'height' => 'integer|min:40|max:200',
            'shape' => 'in:circle,square,rectangle',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $table->update($request->all());
        AuditLog::logUpdate($table, $oldValues);

        return $this->success($table, 'Mesa actualizada');
    }

    /**
     * Change table status
     */
    public function updateTableStatus(Request $request, int $tableId): JsonResponse
    {
        $table = Table::findOrFail($tableId);

        $validator = Validator::make($request->all(), [
            'status' => 'required|in:available,blocked',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        // Can't change status if table is reserved
        if ($table->status === 'reserved') {
            return $this->error('No se puede cambiar el estado de una mesa reservada', 400);
        }

        $table->update(['status' => $request->status]);

        return $this->success($table, 'Estado actualizado');
    }

    /**
     * Delete a table
     */
    public function destroyTable(int $tableId): JsonResponse
    {
        $table = Table::findOrFail($tableId);

        if ($table->reservations()->whereIn('status', ['pending', 'confirmed'])->exists()) {
            return $this->error('No se puede eliminar una mesa con reservas activas', 400);
        }

        AuditLog::logDelete($table);
        $table->delete();

        return $this->success(null, 'Mesa eliminada');
    }

    /**
     * Bulk update table positions (for drag & drop)
     */
    public function updateTablePositions(Request $request, int $roomId): JsonResponse
    {
        $room = Room::findOrFail($roomId);

        $validator = Validator::make($request->all(), [
            'tables' => 'required|array',
            'tables.*.id' => 'required|exists:tables,id',
            'tables.*.position_x' => 'required|integer|min:0',
            'tables.*.position_y' => 'required|integer|min:0',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        foreach ($request->tables as $tableData) {
            Table::where('id', $tableData['id'])
                ->where('room_id', $roomId)
                ->update([
                    'position_x' => $tableData['position_x'],
                    'position_y' => $tableData['position_y'],
                ]);
        }

        return $this->success($room->tables()->get(), 'Posiciones actualizadas');
    }
}
