<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class EventController extends Controller
{
    /**
     * List all events
     */
    public function index(Request $request): JsonResponse
    {
        $query = Event::query();

        // Filters
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        if ($request->has('date_from')) {
            $query->where('start_date', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->where('start_date', '<=', $request->date_to);
        }

        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'start_date');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        $events = $query->paginate($request->get('per_page', 15));

        return $this->paginated($events);
    }

    /**
     * Create a new event
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'start_date' => 'required|date|after_or_equal:today',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'type' => 'required|in:paid,free',
            'ticket_price' => 'required_if:type,paid|numeric|min:0',
            'total_capacity' => 'required|integer|min:1',
            'status' => 'in:draft,active,hidden',
            'room_ids' => 'array',
            'room_ids.*' => 'exists:rooms,id',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $data = $request->except('image', 'room_ids');

        // Handle image upload
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('events', 'public');
            $data['image'] = $path;
        }

        $event = Event::create($data);

        // Attach rooms
        if ($request->has('room_ids')) {
            $event->rooms()->attach($request->room_ids);
        }

        AuditLog::logCreate($event);

        return $this->success($event->fresh(['rooms']), 'Evento creado', 201);
    }

    /**
     * Get event details
     */
    public function show(int $id): JsonResponse
    {
        $event = Event::with(['rooms.tables'])->findOrFail($id);

        return $this->success($event);
    }

    /**
     * Update an event
     */
    public function update(Request $request, int $id): JsonResponse
    {
        $event = Event::findOrFail($id);
        $oldValues = $event->toArray();

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'start_date' => 'sometimes|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'start_time' => 'sometimes|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i',
            'type' => 'sometimes|in:paid,free',
            'ticket_price' => 'nullable|numeric|min:0',
            'total_capacity' => 'sometimes|integer|min:1',
            'status' => 'sometimes|in:draft,active,hidden,finished',
            'room_ids' => 'array',
            'room_ids.*' => 'exists:rooms,id',
        ]);

        if ($validator->fails()) {
            return $this->error('Error de validación', 422, $validator->errors());
        }

        $data = $request->except('image', 'room_ids');

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($event->image) {
                Storage::disk('public')->delete($event->image);
            }
            $path = $request->file('image')->store('events', 'public');
            $data['image'] = $path;
        }

        $event->update($data);

        // Sync rooms
        if ($request->has('room_ids')) {
            $event->rooms()->sync($request->room_ids);
        }

        AuditLog::logUpdate($event, $oldValues);

        return $this->success($event->fresh(['rooms']), 'Evento actualizado');
    }

    /**
     * Delete an event
     */
    public function destroy(int $id): JsonResponse
    {
        $event = Event::findOrFail($id);

        // Check if event has paid orders
        if ($event->orders()->where('status', 'paid')->exists()) {
            return $this->error('No se puede eliminar un evento con órdenes pagadas', 400);
        }

        AuditLog::logDelete($event);

        // Delete image
        if ($event->image) {
            Storage::disk('public')->delete($event->image);
        }

        $event->delete();

        return $this->success(null, 'Evento eliminado');
    }

    /**
     * Get event statistics
     */
    public function stats(int $id): JsonResponse
    {
        $event = Event::findOrFail($id);

        $stats = [
            'total_capacity' => $event->total_capacity,
            'tickets_sold' => $event->tickets_sold,
            'tickets_available' => $event->tickets_available,
            'tickets_used' => $event->tickets()->where('status', 'used')->count(),
            'total_revenue' => $event->total_revenue,
            'orders_count' => $event->orders()->where('status', 'paid')->count(),
            'sales_by_channel' => [
                'web' => $event->orders()->where('status', 'paid')->where('channel', 'web')->sum('total'),
                'tpv' => $event->orders()->where('status', 'paid')->where('channel', 'tpv')->sum('total'),
            ],
            'occupancy_rate' => $event->total_capacity > 0
                ? round(($event->tickets_sold / $event->total_capacity) * 100, 2)
                : 0,
        ];

        return $this->success($stats);
    }
}
