<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Event;
use App\Models\Order;
use App\Models\Ticket;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Get dashboard statistics
     */
    public function index(): JsonResponse
    {
        $today = now()->startOfDay();
        $startOfWeek = now()->startOfWeek();
        $startOfMonth = now()->startOfMonth();

        // Sales statistics
        $salesStats = [
            'today' => Order::paid()->whereDate('created_at', $today)->sum('total'),
            'week' => Order::paid()->where('created_at', '>=', $startOfWeek)->sum('total'),
            'month' => Order::paid()->where('created_at', '>=', $startOfMonth)->sum('total'),
        ];

        // Tickets statistics
        $ticketsStats = [
            'sold_today' => Ticket::whereDate('created_at', $today)->count(),
            'sold_week' => Ticket::where('created_at', '>=', $startOfWeek)->count(),
            'used_today' => Ticket::where('status', 'used')->whereDate('used_at', $today)->count(),
        ];

        // Active events
        $activeEvents = Event::active()
            ->upcoming()
            ->withCount([
                'tickets as tickets_sold' => function ($query) {
                    $query->whereIn('status', ['valid', 'used']);
                }
            ])
            ->orderBy('start_date')
            ->limit(5)
            ->get();

        // Recent orders
        $recentOrders = Order::with(['user:id,name,email', 'event:id,name'])
            ->latest()
            ->limit(10)
            ->get();

        // Sales by channel
        $salesByChannel = Order::paid()
            ->where('created_at', '>=', $startOfMonth)
            ->select('channel', DB::raw('SUM(total) as total'), DB::raw('COUNT(*) as count'))
            ->groupBy('channel')
            ->get();

        // Daily sales for chart (last 7 days)
        $dailySales = Order::paid()
            ->where('created_at', '>=', now()->subDays(7))
            ->select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('SUM(total) as total'),
                DB::raw('COUNT(*) as orders')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Alerts
        $alerts = [];

        // Events running low on capacity
        $lowCapacityEvents = Event::active()
            ->upcoming()
            ->get()
            ->filter(function ($event) {
                return $event->tickets_available > 0 && $event->tickets_available <= 10;
            });

        foreach ($lowCapacityEvents as $event) {
            $alerts[] = [
                'type' => 'warning',
                'message' => "Evento '{$event->name}' casi lleno: {$event->tickets_available} entradas disponibles",
            ];
        }

        // Pending orders
        $pendingOrders = Order::pending()->where('created_at', '<=', now()->subHours(1))->count();
        if ($pendingOrders > 0) {
            $alerts[] = [
                'type' => 'info',
                'message' => "{$pendingOrders} órdenes pendientes hace más de 1 hora",
            ];
        }

        return $this->success([
            'sales' => $salesStats,
            'tickets' => $ticketsStats,
            'active_events' => $activeEvents,
            'recent_orders' => $recentOrders,
            'sales_by_channel' => $salesByChannel,
            'daily_sales' => $dailySales,
            'alerts' => $alerts,
            'counts' => [
                'total_clients' => User::where('role', 'client')->count(),
                'active_events' => Event::active()->count(),
                'orders_today' => Order::whereDate('created_at', $today)->count(),
            ],
        ]);
    }
}
