import axios from 'axios';

const api = axios.create({
    baseURL: '/api',
    headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
    },
});

// Request interceptor - add auth token
api.interceptors.request.use(
    (config) => {
        const token = localStorage.getItem('token');
        if (token) {
            config.headers.Authorization = `Bearer ${token}`;
        }
        return config;
    },
    (error) => Promise.reject(error)
);

// Response interceptor - handle errors
api.interceptors.response.use(
    (response) => response,
    (error) => {
        if (error.response?.status === 401) {
            localStorage.removeItem('token');
            localStorage.removeItem('user');
            window.location.href = '/login';
        }
        return Promise.reject(error);
    }
);

// Auth API
export const authAPI = {
    login: (email, password) => api.post('/auth/login', { email, password }),
    logout: () => api.post('/auth/logout'),
    me: () => api.get('/auth/me'),
    updateProfile: (data) => api.put('/auth/profile', data),
};

// Dashboard API
export const dashboardAPI = {
    getStats: () => api.get('/admin/dashboard'),
};

// Events API
export const eventsAPI = {
    list: (params) => api.get('/admin/events', { params }),
    get: (id) => api.get(`/admin/events/${id}`),
    create: (data) => api.post('/admin/events', data),
    update: (id, data) => api.put(`/admin/events/${id}`, data),
    delete: (id) => api.delete(`/admin/events/${id}`),
    stats: (id) => api.get(`/admin/events/${id}/stats`),
};

// Rooms API
export const roomsAPI = {
    list: () => api.get('/admin/rooms'),
    get: (id) => api.get(`/admin/rooms/${id}`),
    create: (data) => api.post('/admin/rooms', data),
    update: (id, data) => api.put(`/admin/rooms/${id}`, data),
    delete: (id) => api.delete(`/admin/rooms/${id}`),
    tables: (roomId) => api.get(`/admin/rooms/${roomId}/tables`),
    createTable: (roomId, data) => api.post(`/admin/rooms/${roomId}/tables`, data),
    updateTable: (tableId, data) => api.put(`/admin/tables/${tableId}`, data),
    updateTableStatus: (tableId, status) => api.patch(`/admin/tables/${tableId}/status`, { status }),
    deleteTable: (tableId) => api.delete(`/admin/tables/${tableId}`),
    updateTablePositions: (roomId, tables) => api.put(`/admin/rooms/${roomId}/tables/positions`, { tables }),
};

// Clients API
export const clientsAPI = {
    list: (params) => api.get('/admin/clients', { params }),
    get: (id) => api.get(`/admin/clients/${id}`),
};

// Settings API
export const settingsAPI = {
    get: () => api.get('/admin/settings'),
    update: (data) => api.put('/admin/settings', data),
};

// Users API
export const usersAPI = {
    list: () => api.get('/admin/users'),
    create: (data) => api.post('/admin/users', data),
};

// Logs API
export const logsAPI = {
    list: (params) => api.get('/admin/logs', { params }),
};

export default api;
