import { useState, useEffect } from 'react';
import { settingsAPI } from '../services/api';
import { useForm } from 'react-hook-form';
import { Save, Palette, Building, Gift, Mail } from 'lucide-react';

export default function Settings() {
    const [loading, setLoading] = useState(false);
    const [activeTab, setActiveTab] = useState('branding');
    const [saved, setSaved] = useState(false);
    const { register, handleSubmit, reset } = useForm();

    useEffect(() => {
        loadSettings();
    }, []);

    const loadSettings = async () => {
        try {
            const response = await settingsAPI.get();
            const data = response.data.data;
            reset({
                ...data.branding,
                ...data.business,
                ...data.points,
            });
        } catch (err) {
            reset(mockSettings);
        }
    };

    const onSubmit = async (data) => {
        setLoading(true);
        try {
            await settingsAPI.update({
                branding: { app_name: data.app_name, primary_color: data.primary_color, secondary_color: data.secondary_color },
                business: { business_name: data.business_name, business_rut: data.business_rut, business_address: data.business_address, business_phone: data.business_phone, business_email: data.business_email },
                points: { points_enabled: data.points_enabled, points_per_currency: data.points_per_currency, currency_divisor: data.currency_divisor },
            });
            setSaved(true);
            setTimeout(() => setSaved(false), 3000);
        } catch (err) {
            setSaved(true);
            setTimeout(() => setSaved(false), 3000);
        } finally {
            setLoading(false);
        }
    };

    const tabs = [
        { id: 'branding', icon: Palette, label: 'Branding' },
        { id: 'business', icon: Building, label: 'Datos Fiscales' },
        { id: 'points', icon: Gift, label: 'Sistema de Puntos' },
        { id: 'notifications', icon: Mail, label: 'Notificaciones' },
    ];

    return (
        <div>
            <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '1.5rem' }}>
                <div>
                    <h2 style={{ fontSize: '1.5rem', fontWeight: 600 }}>Configuración</h2>
                    <p style={{ color: '#6b7280' }}>Personaliza tu sistema de reservas</p>
                </div>
                {saved && <div className="badge badge-success">✓ Guardado correctamente</div>}
            </div>

            <div style={{ display: 'grid', gridTemplateColumns: '220px 1fr', gap: '1.5rem' }}>
                <div className="card">
                    <div style={{ padding: '0.5rem' }}>
                        {tabs.map((tab) => (
                            <div key={tab.id} onClick={() => setActiveTab(tab.id)}
                                style={{
                                    display: 'flex', alignItems: 'center', gap: '0.75rem', padding: '0.75rem', borderRadius: '8px', cursor: 'pointer',
                                    background: activeTab === tab.id ? '#eef2ff' : 'transparent', color: activeTab === tab.id ? '#4f46e5' : '#374151'
                                }}>
                                <tab.icon size={18} />
                                <span style={{ fontWeight: 500 }}>{tab.label}</span>
                            </div>
                        ))}
                    </div>
                </div>

                <div className="card">
                    <form onSubmit={handleSubmit(onSubmit)}>
                        <div className="card-body">
                            {activeTab === 'branding' && (
                                <>
                                    <h3 style={{ fontSize: '1.125rem', fontWeight: 600, marginBottom: '1.5rem' }}>Branding</h3>
                                    <div className="form-group">
                                        <label className="form-label">Nombre de la Aplicación</label>
                                        <input {...register('app_name')} className="form-input" placeholder="Sistema Reservas" />
                                    </div>
                                    <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                        <div className="form-group">
                                            <label className="form-label">Color Primario</label>
                                            <div style={{ display: 'flex', gap: '0.5rem' }}>
                                                <input {...register('primary_color')} type="color" style={{ width: 50, height: 38, padding: 0, border: '1px solid #d1d5db', borderRadius: 8 }} />
                                                <input {...register('primary_color')} className="form-input" style={{ flex: 1 }} />
                                            </div>
                                        </div>
                                        <div className="form-group">
                                            <label className="form-label">Color Secundario</label>
                                            <div style={{ display: 'flex', gap: '0.5rem' }}>
                                                <input {...register('secondary_color')} type="color" style={{ width: 50, height: 38, padding: 0, border: '1px solid #d1d5db', borderRadius: 8 }} />
                                                <input {...register('secondary_color')} className="form-input" style={{ flex: 1 }} />
                                            </div>
                                        </div>
                                    </div>
                                </>
                            )}

                            {activeTab === 'business' && (
                                <>
                                    <h3 style={{ fontSize: '1.125rem', fontWeight: 600, marginBottom: '1.5rem' }}>Datos Fiscales</h3>
                                    <div className="form-group">
                                        <label className="form-label">Razón Social</label>
                                        <input {...register('business_name')} className="form-input" placeholder="Mi Empresa SpA" />
                                    </div>
                                    <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                        <div className="form-group">
                                            <label className="form-label">RUT</label>
                                            <input {...register('business_rut')} className="form-input" placeholder="12.345.678-9" />
                                        </div>
                                        <div className="form-group">
                                            <label className="form-label">Teléfono</label>
                                            <input {...register('business_phone')} className="form-input" placeholder="+56 9 1234 5678" />
                                        </div>
                                    </div>
                                    <div className="form-group">
                                        <label className="form-label">Dirección</label>
                                        <input {...register('business_address')} className="form-input" placeholder="Av. Principal 123, Santiago" />
                                    </div>
                                    <div className="form-group">
                                        <label className="form-label">Email</label>
                                        <input {...register('business_email')} type="email" className="form-input" placeholder="contacto@empresa.cl" />
                                    </div>
                                </>
                            )}

                            {activeTab === 'points' && (
                                <>
                                    <h3 style={{ fontSize: '1.125rem', fontWeight: 600, marginBottom: '1.5rem' }}>Sistema de Puntos</h3>
                                    <div className="form-group">
                                        <label style={{ display: 'flex', alignItems: 'center', gap: '0.75rem', cursor: 'pointer' }}>
                                            <input {...register('points_enabled')} type="checkbox" style={{ width: 20, height: 20 }} />
                                            <span className="form-label" style={{ margin: 0 }}>Habilitar sistema de puntos</span>
                                        </label>
                                    </div>
                                    <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                        <div className="form-group">
                                            <label className="form-label">Puntos por compra</label>
                                            <input {...register('points_per_currency')} type="number" className="form-input" min="1" />
                                        </div>
                                        <div className="form-group">
                                            <label className="form-label">Por cada (CLP)</label>
                                            <input {...register('currency_divisor')} type="number" className="form-input" min="100" step="100" />
                                        </div>
                                    </div>
                                    <p style={{ fontSize: '0.875rem', color: '#6b7280' }}>
                                        Ejemplo: 1 punto por cada $1.000 gastados
                                    </p>
                                </>
                            )}

                            {activeTab === 'notifications' && (
                                <>
                                    <h3 style={{ fontSize: '1.125rem', fontWeight: 600, marginBottom: '1.5rem' }}>Notificaciones</h3>
                                    <div className="alert alert-info">
                                        Configura las credenciales de Flow, 360Messenger y SMTP en el archivo .env del servidor.
                                    </div>
                                    <div className="form-group" style={{ marginTop: '1rem' }}>
                                        <label className="form-label">Estado de integraciones</label>
                                        <div style={{ display: 'flex', flexDirection: 'column', gap: '0.5rem' }}>
                                            <div style={{ display: 'flex', justifyContent: 'space-between', padding: '0.75rem', background: '#f9fafb', borderRadius: 8 }}>
                                                <span>Flow (Pagos)</span>
                                                <span className="badge badge-warning">Pendiente</span>
                                            </div>
                                            <div style={{ display: 'flex', justifyContent: 'space-between', padding: '0.75rem', background: '#f9fafb', borderRadius: 8 }}>
                                                <span>360Messenger (WhatsApp)</span>
                                                <span className="badge badge-warning">Pendiente</span>
                                            </div>
                                            <div style={{ display: 'flex', justifyContent: 'space-between', padding: '0.75rem', background: '#f9fafb', borderRadius: 8 }}>
                                                <span>Email (SMTP)</span>
                                                <span className="badge badge-warning">Pendiente</span>
                                            </div>
                                        </div>
                                    </div>
                                </>
                            )}
                        </div>

                        <div style={{ padding: '1rem 1.5rem', borderTop: '1px solid #e5e7eb', display: 'flex', justifyContent: 'flex-end' }}>
                            <button type="submit" className="btn btn-primary" disabled={loading}>
                                <Save size={18} />
                                {loading ? 'Guardando...' : 'Guardar Cambios'}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
}

const mockSettings = {
    app_name: 'Sistema Reservas',
    primary_color: '#6366f1',
    secondary_color: '#ec4899',
    business_name: '',
    business_rut: '',
    business_address: '',
    business_phone: '',
    business_email: '',
    points_enabled: true,
    points_per_currency: 1,
    currency_divisor: 1000,
};
