import { useState, useEffect } from 'react';
import { roomsAPI } from '../services/api';
import { useForm } from 'react-hook-form';
import { Plus, Edit, X, Building2, Users, Lock, Unlock } from 'lucide-react';

export default function Rooms() {
    const [rooms, setRooms] = useState(mockRooms);
    const [selectedRoom, setSelectedRoom] = useState(mockRooms[0]);
    const [loading, setLoading] = useState(false);
    const [showTableModal, setShowTableModal] = useState(false);
    const [editingTable, setEditingTable] = useState(null);
    const tableForm = useForm();

    const openTableModal = (table = null) => {
        setEditingTable(table);
        tableForm.reset(table || { name: '', capacity: 4, position_x: 50, position_y: 50, shape: 'circle' });
        setShowTableModal(true);
    };

    const getTableStyle = (table) => ({
        position: 'absolute',
        left: `${table.position_x}px`,
        top: `${table.position_y}px`,
        width: `${table.width || 80}px`,
        height: `${table.height || 80}px`,
        borderRadius: table.shape === 'circle' ? '50%' : '8px',
        background: table.status === 'available' ? 'linear-gradient(135deg, #10b981, #059669)'
            : table.status === 'reserved' ? 'linear-gradient(135deg, #f59e0b, #d97706)'
                : 'linear-gradient(135deg, #6b7280, #4b5563)',
        display: 'flex', flexDirection: 'column', alignItems: 'center', justifyContent: 'center',
        color: 'white', fontSize: '0.75rem', fontWeight: 500, cursor: 'pointer',
        boxShadow: '0 4px 6px -1px rgba(0,0,0,0.1)', transition: 'transform 0.2s',
    });

    return (
        <div>
            <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '1.5rem' }}>
                <div>
                    <h2 style={{ fontSize: '1.5rem', fontWeight: 600 }}>Salones y Mesas</h2>
                    <p style={{ color: '#6b7280' }}>Configura la distribución de mesas</p>
                </div>
                <button className="btn btn-primary"><Plus size={18} /> Nuevo Salón</button>
            </div>

            <div style={{ display: 'grid', gridTemplateColumns: '280px 1fr', gap: '1.5rem', minHeight: '500px' }}>
                <div className="card">
                    <div className="card-header"><h3 className="card-title">Salones</h3></div>
                    <div style={{ padding: '0.5rem' }}>
                        {rooms.map((room) => (
                            <div key={room.id} onClick={() => setSelectedRoom(room)}
                                style={{
                                    display: 'flex', alignItems: 'center', gap: '0.75rem', padding: '0.75rem', borderRadius: '8px', cursor: 'pointer',
                                    background: selectedRoom?.id === room.id ? '#eef2ff' : 'transparent'
                                }}>
                                <Building2 size={18} />
                                <div style={{ flex: 1 }}>
                                    <div style={{ fontWeight: 500 }}>{room.name}</div>
                                    <div style={{ fontSize: '0.75rem', color: '#6b7280' }}>{room.tables?.length || 0} mesas</div>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>

                <div className="card">
                    <div className="card-header">
                        <h3 className="card-title">{selectedRoom?.name}</h3>
                        <button className="btn btn-sm btn-primary" onClick={() => openTableModal()}>
                            <Plus size={16} /> Mesa
                        </button>
                    </div>
                    <div className="card-body">
                        <div style={{ display: 'flex', gap: '1.5rem', marginBottom: '1rem' }}>
                            {[['#10b981', 'Disponible'], ['#f59e0b', 'Reservada'], ['#6b7280', 'Bloqueada']].map(([color, label]) => (
                                <div key={label} style={{ display: 'flex', alignItems: 'center', gap: '0.5rem', fontSize: '0.875rem', color: '#6b7280' }}>
                                    <div style={{ width: 12, height: 12, borderRadius: '50%', background: color }}></div>
                                    <span>{label}</span>
                                </div>
                            ))}
                        </div>
                        <div style={{ position: 'relative', height: '400px', background: '#f9fafb', borderRadius: '8px', border: '2px dashed #e5e7eb' }}>
                            {selectedRoom?.tables?.map((table) => (
                                <div key={table.id} style={getTableStyle(table)} onClick={() => openTableModal(table)}>
                                    <span style={{ fontWeight: 600 }}>{table.name}</span>
                                    <span style={{ fontSize: '0.625rem' }}><Users size={10} /> {table.capacity}</span>
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
            </div>

            {showTableModal && (
                <div className="modal-overlay" onClick={() => setShowTableModal(false)}>
                    <div className="modal" onClick={(e) => e.stopPropagation()}>
                        <div className="modal-header">
                            <h3 className="modal-title">{editingTable ? 'Editar Mesa' : 'Nueva Mesa'}</h3>
                            <button className="modal-close" onClick={() => setShowTableModal(false)}><X size={20} /></button>
                        </div>
                        <form onSubmit={tableForm.handleSubmit(() => setShowTableModal(false))}>
                            <div className="modal-body">
                                <div className="form-group">
                                    <label className="form-label">Nombre</label>
                                    <input {...tableForm.register('name')} className="form-input" placeholder="Mesa 1" />
                                </div>
                                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                    <div className="form-group">
                                        <label className="form-label">Capacidad</label>
                                        <input {...tableForm.register('capacity')} type="number" className="form-input" min="1" max="20" />
                                    </div>
                                    <div className="form-group">
                                        <label className="form-label">Forma</label>
                                        <select {...tableForm.register('shape')} className="form-select">
                                            <option value="circle">Círculo</option>
                                            <option value="square">Cuadrado</option>
                                            <option value="rectangle">Rectángulo</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            <div className="modal-footer">
                                <button type="button" className="btn btn-secondary" onClick={() => setShowTableModal(false)}>Cancelar</button>
                                <button type="submit" className="btn btn-primary">Guardar</button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}

const mockRooms = [
    {
        id: 1, name: 'Salón Principal', capacity: 100, tables: [
            { id: 1, name: 'Mesa 1', capacity: 4, position_x: 50, position_y: 50, width: 80, height: 80, shape: 'circle', status: 'available' },
            { id: 2, name: 'Mesa 2', capacity: 4, position_x: 160, position_y: 50, width: 80, height: 80, shape: 'circle', status: 'reserved' },
            { id: 3, name: 'Mesa 3', capacity: 6, position_x: 270, position_y: 50, width: 100, height: 80, shape: 'rectangle', status: 'available' },
            { id: 4, name: 'Mesa 4', capacity: 4, position_x: 50, position_y: 160, width: 80, height: 80, shape: 'circle', status: 'blocked' },
            { id: 5, name: 'Mesa 5', capacity: 8, position_x: 160, position_y: 160, width: 120, height: 80, shape: 'rectangle', status: 'available' },
        ]
    },
    {
        id: 2, name: 'Terraza VIP', capacity: 40, tables: [
            { id: 6, name: 'VIP 1', capacity: 6, position_x: 50, position_y: 50, width: 90, height: 90, shape: 'square', status: 'available' },
            { id: 7, name: 'VIP 2', capacity: 6, position_x: 170, position_y: 50, width: 90, height: 90, shape: 'square', status: 'available' },
        ]
    },
];
