import { useState, useEffect } from 'react';
import { eventsAPI } from '../services/api';
import { useForm } from 'react-hook-form';
import {
    Plus,
    Search,
    Edit,
    Trash2,
    Calendar,
    Users,
    DollarSign,
    X,
    Eye,
    BarChart2,
} from 'lucide-react';

export default function Events() {
    const [events, setEvents] = useState([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [editingEvent, setEditingEvent] = useState(null);
    const [search, setSearch] = useState('');
    const [statusFilter, setStatusFilter] = useState('');

    const { register, handleSubmit, reset, formState: { errors } } = useForm();

    useEffect(() => {
        loadEvents();
    }, [statusFilter]);

    const loadEvents = async () => {
        setLoading(true);
        try {
            const params = {};
            if (statusFilter) params.status = statusFilter;
            if (search) params.search = search;

            const response = await eventsAPI.list(params);
            setEvents(response.data.data || mockEvents);
        } catch (err) {
            setEvents(mockEvents);
        } finally {
            setLoading(false);
        }
    };

    const openCreateModal = () => {
        setEditingEvent(null);
        reset({
            name: '',
            description: '',
            start_date: '',
            start_time: '',
            end_time: '',
            type: 'paid',
            ticket_price: 0,
            total_capacity: 100,
            status: 'draft',
        });
        setShowModal(true);
    };

    const openEditModal = (event) => {
        setEditingEvent(event);
        reset({
            name: event.name,
            description: event.description,
            start_date: event.start_date,
            start_time: event.start_time,
            end_time: event.end_time,
            type: event.type,
            ticket_price: event.ticket_price,
            total_capacity: event.total_capacity,
            status: event.status,
        });
        setShowModal(true);
    };

    const onSubmit = async (data) => {
        try {
            if (editingEvent) {
                await eventsAPI.update(editingEvent.id, data);
            } else {
                await eventsAPI.create(data);
            }
            setShowModal(false);
            loadEvents();
        } catch (err) {
            alert('Error al guardar el evento');
        }
    };

    const handleDelete = async (id) => {
        if (!confirm('¿Estás seguro de eliminar este evento?')) return;
        try {
            await eventsAPI.delete(id);
            loadEvents();
        } catch (err) {
            alert('Error al eliminar el evento');
        }
    };

    const formatDate = (date) => {
        return new Date(date).toLocaleDateString('es-CL', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
        });
    };

    const formatCurrency = (value) => {
        return new Intl.NumberFormat('es-CL', {
            style: 'currency',
            currency: 'CLP',
            minimumFractionDigits: 0,
        }).format(value);
    };

    return (
        <div>
            {/* Header */}
            <div style={styles.header}>
                <div>
                    <h2 style={{ fontSize: '1.5rem', fontWeight: 600, marginBottom: '0.25rem' }}>
                        Gestión de Eventos
                    </h2>
                    <p style={{ color: '#6b7280' }}>
                        Administra tus eventos, entradas y capacidad
                    </p>
                </div>
                <button className="btn btn-primary" onClick={openCreateModal}>
                    <Plus size={18} />
                    Nuevo Evento
                </button>
            </div>

            {/* Filters */}
            <div className="card" style={{ marginBottom: '1.5rem' }}>
                <div className="card-body" style={{ display: 'flex', gap: '1rem', flexWrap: 'wrap' }}>
                    <div style={{ flex: 1, minWidth: '200px' }}>
                        <div style={{ position: 'relative' }}>
                            <Search
                                size={18}
                                style={{ position: 'absolute', left: '0.75rem', top: '50%', transform: 'translateY(-50%)', color: '#9ca3af' }}
                            />
                            <input
                                type="text"
                                className="form-input"
                                placeholder="Buscar evento..."
                                value={search}
                                onChange={(e) => setSearch(e.target.value)}
                                style={{ paddingLeft: '2.5rem' }}
                            />
                        </div>
                    </div>
                    <select
                        className="form-select"
                        value={statusFilter}
                        onChange={(e) => setStatusFilter(e.target.value)}
                        style={{ width: '180px' }}
                    >
                        <option value="">Todos los estados</option>
                        <option value="active">Activo</option>
                        <option value="draft">Borrador</option>
                        <option value="hidden">Oculto</option>
                        <option value="finished">Finalizado</option>
                    </select>
                    <button className="btn btn-secondary" onClick={loadEvents}>
                        Buscar
                    </button>
                </div>
            </div>

            {/* Events Grid */}
            {loading ? (
                <div style={{ display: 'flex', justifyContent: 'center', padding: '3rem' }}>
                    <div className="loading-spinner"></div>
                </div>
            ) : (
                <div style={styles.grid}>
                    {events.map((event) => (
                        <div key={event.id} className="card" style={styles.eventCard}>
                            <div style={styles.eventImage}>
                                {event.image ? (
                                    <img src={event.image} alt={event.name} style={{ width: '100%', height: '100%', objectFit: 'cover' }} />
                                ) : (
                                    <Calendar size={48} style={{ color: '#d1d5db' }} />
                                )}
                                <span className={`badge badge-${event.status === 'active' ? 'success' : event.status === 'draft' ? 'warning' : 'gray'}`} style={styles.statusBadge}>
                                    {event.status === 'active' ? 'Activo' : event.status === 'draft' ? 'Borrador' : event.status}
                                </span>
                            </div>

                            <div style={styles.eventContent}>
                                <h3 style={styles.eventTitle}>{event.name}</h3>

                                <div style={styles.eventMeta}>
                                    <div style={styles.metaItem}>
                                        <Calendar size={14} />
                                        <span>{formatDate(event.start_date)}</span>
                                    </div>
                                    <div style={styles.metaItem}>
                                        <Users size={14} />
                                        <span>{event.tickets_sold || 0} / {event.total_capacity}</span>
                                    </div>
                                </div>

                                <div style={styles.eventPrice}>
                                    {event.type === 'free' ? (
                                        <span className="badge badge-success">Gratis</span>
                                    ) : (
                                        <span style={{ fontWeight: 600, color: '#111827' }}>
                                            {formatCurrency(event.ticket_price)}
                                        </span>
                                    )}
                                </div>

                                <div style={styles.eventActions}>
                                    <button className="btn btn-sm btn-secondary" onClick={() => openEditModal(event)}>
                                        <Edit size={14} />
                                        Editar
                                    </button>
                                    <button className="btn btn-sm btn-secondary">
                                        <BarChart2 size={14} />
                                    </button>
                                    <button className="btn btn-sm btn-danger" onClick={() => handleDelete(event.id)}>
                                        <Trash2 size={14} />
                                    </button>
                                </div>
                            </div>
                        </div>
                    ))}
                </div>
            )}

            {/* Modal */}
            {showModal && (
                <div className="modal-overlay" onClick={() => setShowModal(false)}>
                    <div className="modal" onClick={(e) => e.stopPropagation()} style={{ maxWidth: '600px' }}>
                        <div className="modal-header">
                            <h3 className="modal-title">
                                {editingEvent ? 'Editar Evento' : 'Nuevo Evento'}
                            </h3>
                            <button className="modal-close" onClick={() => setShowModal(false)}>
                                <X size={20} />
                            </button>
                        </div>

                        <form onSubmit={handleSubmit(onSubmit)}>
                            <div className="modal-body">
                                <div className="form-group">
                                    <label className="form-label">Nombre del Evento *</label>
                                    <input
                                        {...register('name', { required: 'El nombre es requerido' })}
                                        className="form-input"
                                        placeholder="Ej: Noche de Inauguración"
                                    />
                                    {errors.name && <span className="form-error">{errors.name.message}</span>}
                                </div>

                                <div className="form-group">
                                    <label className="form-label">Descripción</label>
                                    <textarea
                                        {...register('description')}
                                        className="form-textarea"
                                        placeholder="Descripción del evento..."
                                        rows={3}
                                    />
                                </div>

                                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                    <div className="form-group">
                                        <label className="form-label">Fecha *</label>
                                        <input
                                            {...register('start_date', { required: true })}
                                            type="date"
                                            className="form-input"
                                        />
                                    </div>
                                    <div className="form-group">
                                        <label className="form-label">Hora Inicio *</label>
                                        <input
                                            {...register('start_time', { required: true })}
                                            type="time"
                                            className="form-input"
                                        />
                                    </div>
                                </div>

                                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                    <div className="form-group">
                                        <label className="form-label">Tipo de Evento</label>
                                        <select {...register('type')} className="form-select">
                                            <option value="paid">🎟 Pago</option>
                                            <option value="free">🆓 Gratis</option>
                                        </select>
                                    </div>
                                    <div className="form-group">
                                        <label className="form-label">Precio Entrada (CLP)</label>
                                        <input
                                            {...register('ticket_price')}
                                            type="number"
                                            className="form-input"
                                            min="0"
                                            step="500"
                                        />
                                    </div>
                                </div>

                                <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                                    <div className="form-group">
                                        <label className="form-label">Capacidad Total</label>
                                        <input
                                            {...register('total_capacity')}
                                            type="number"
                                            className="form-input"
                                            min="1"
                                        />
                                    </div>
                                    <div className="form-group">
                                        <label className="form-label">Estado</label>
                                        <select {...register('status')} className="form-select">
                                            <option value="draft">Borrador</option>
                                            <option value="active">Activo</option>
                                            <option value="hidden">Oculto</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <div className="modal-footer">
                                <button type="button" className="btn btn-secondary" onClick={() => setShowModal(false)}>
                                    Cancelar
                                </button>
                                <button type="submit" className="btn btn-primary">
                                    {editingEvent ? 'Guardar Cambios' : 'Crear Evento'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}

const styles = {
    header: {
        display: 'flex',
        justifyContent: 'space-between',
        alignItems: 'center',
        marginBottom: '1.5rem',
        flexWrap: 'wrap',
        gap: '1rem',
    },
    grid: {
        display: 'grid',
        gridTemplateColumns: 'repeat(auto-fill, minmax(320px, 1fr))',
        gap: '1.5rem',
    },
    eventCard: {
        overflow: 'hidden',
    },
    eventImage: {
        height: '140px',
        background: 'linear-gradient(135deg, #f3f4f6, #e5e7eb)',
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        position: 'relative',
    },
    statusBadge: {
        position: 'absolute',
        top: '0.75rem',
        right: '0.75rem',
    },
    eventContent: {
        padding: '1.25rem',
    },
    eventTitle: {
        fontSize: '1.125rem',
        fontWeight: 600,
        marginBottom: '0.75rem',
    },
    eventMeta: {
        display: 'flex',
        gap: '1rem',
        marginBottom: '0.75rem',
    },
    metaItem: {
        display: 'flex',
        alignItems: 'center',
        gap: '0.375rem',
        fontSize: '0.875rem',
        color: '#6b7280',
    },
    eventPrice: {
        marginBottom: '1rem',
    },
    eventActions: {
        display: 'flex',
        gap: '0.5rem',
    },
};

const mockEvents = [
    { id: 1, name: 'Noche de Inauguración', start_date: '2024-01-15', start_time: '21:00', type: 'paid', ticket_price: 15000, total_capacity: 200, tickets_sold: 145, status: 'active' },
    { id: 2, name: 'After Office Viernes', start_date: '2024-01-12', start_time: '18:00', type: 'free', ticket_price: 0, total_capacity: 150, tickets_sold: 89, status: 'active' },
    { id: 3, name: 'Fiesta Retro 80s', start_date: '2024-01-20', start_time: '22:00', type: 'paid', ticket_price: 12000, total_capacity: 180, tickets_sold: 0, status: 'draft' },
];
