import { useState, useEffect } from 'react';
import { dashboardAPI } from '../services/api';
import {
    DollarSign,
    Ticket,
    Users,
    Calendar,
    TrendingUp,
    TrendingDown,
    AlertCircle,
} from 'lucide-react';
import {
    LineChart,
    Line,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
    BarChart,
    Bar,
} from 'recharts';

export default function Dashboard() {
    const [data, setData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    useEffect(() => {
        loadDashboard();
    }, []);

    const loadDashboard = async () => {
        try {
            const response = await dashboardAPI.getStats();
            setData(response.data.data);
        } catch (err) {
            setError('Error al cargar el dashboard');
            // Use mock data for demo
            setData(mockData);
        } finally {
            setLoading(false);
        }
    };

    if (loading) {
        return (
            <div style={{ display: 'flex', justifyContent: 'center', padding: '3rem' }}>
                <div className="loading-spinner"></div>
            </div>
        );
    }

    const formatCurrency = (value) => {
        return new Intl.NumberFormat('es-CL', {
            style: 'currency',
            currency: 'CLP',
            minimumFractionDigits: 0,
        }).format(value);
    };

    return (
        <div>
            <div style={{ marginBottom: '1.5rem' }}>
                <h2 style={{ fontSize: '1.5rem', fontWeight: 600, marginBottom: '0.25rem' }}>
                    Dashboard
                </h2>
                <p style={{ color: '#6b7280' }}>
                    Resumen de ventas y actividad del sistema
                </p>
            </div>

            {/* Stat Cards */}
            <div className="stats-grid">
                <div className="stat-card">
                    <div className="stat-icon primary">
                        <DollarSign size={24} />
                    </div>
                    <div className="stat-content">
                        <div className="stat-label">Ventas Hoy</div>
                        <div className="stat-value">{formatCurrency(data?.sales?.today || 0)}</div>
                        <div className="stat-change positive">
                            <TrendingUp size={14} style={{ marginRight: '0.25rem' }} />
                            +12% vs ayer
                        </div>
                    </div>
                </div>

                <div className="stat-card">
                    <div className="stat-icon success">
                        <Ticket size={24} />
                    </div>
                    <div className="stat-content">
                        <div className="stat-label">Entradas Vendidas</div>
                        <div className="stat-value">{data?.tickets?.sold_today || 0}</div>
                        <div className="stat-change positive">
                            <TrendingUp size={14} style={{ marginRight: '0.25rem' }} />
                            {data?.tickets?.sold_week || 0} esta semana
                        </div>
                    </div>
                </div>

                <div className="stat-card">
                    <div className="stat-icon warning">
                        <Users size={24} />
                    </div>
                    <div className="stat-content">
                        <div className="stat-label">Clientes</div>
                        <div className="stat-value">{data?.counts?.total_clients || 0}</div>
                        <div className="stat-label" style={{ marginTop: '0.25rem' }}>
                            Registrados
                        </div>
                    </div>
                </div>

                <div className="stat-card">
                    <div className="stat-icon accent">
                        <Calendar size={24} />
                    </div>
                    <div className="stat-content">
                        <div className="stat-label">Eventos Activos</div>
                        <div className="stat-value">{data?.counts?.active_events || 0}</div>
                        <div className="stat-label" style={{ marginTop: '0.25rem' }}>
                            Próximos eventos
                        </div>
                    </div>
                </div>
            </div>

            {/* Charts Row */}
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(400px, 1fr))', gap: '1.5rem', marginBottom: '1.5rem' }}>
                {/* Sales Chart */}
                <div className="card">
                    <div className="card-header">
                        <h3 className="card-title">Ventas Últimos 7 Días</h3>
                    </div>
                    <div className="card-body">
                        <ResponsiveContainer width="100%" height={250}>
                            <LineChart data={data?.daily_sales || mockData.daily_sales}>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                                <XAxis
                                    dataKey="date"
                                    stroke="#9ca3af"
                                    fontSize={12}
                                    tickFormatter={(val) => val.slice(5)}
                                />
                                <YAxis
                                    stroke="#9ca3af"
                                    fontSize={12}
                                    tickFormatter={(val) => `$${val / 1000}k`}
                                />
                                <Tooltip
                                    formatter={(value) => [formatCurrency(value), 'Ventas']}
                                    contentStyle={{
                                        background: 'white',
                                        border: '1px solid #e5e7eb',
                                        borderRadius: '8px',
                                        boxShadow: '0 4px 6px -1px rgba(0,0,0,0.1)',
                                    }}
                                />
                                <Line
                                    type="monotone"
                                    dataKey="total"
                                    stroke="#6366f1"
                                    strokeWidth={2}
                                    dot={{ fill: '#6366f1', strokeWidth: 2 }}
                                    activeDot={{ r: 6 }}
                                />
                            </LineChart>
                        </ResponsiveContainer>
                    </div>
                </div>

                {/* Sales by Channel */}
                <div className="card">
                    <div className="card-header">
                        <h3 className="card-title">Ventas por Canal</h3>
                    </div>
                    <div className="card-body">
                        <ResponsiveContainer width="100%" height={250}>
                            <BarChart data={data?.sales_by_channel || mockData.sales_by_channel}>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                                <XAxis dataKey="channel" stroke="#9ca3af" fontSize={12} />
                                <YAxis stroke="#9ca3af" fontSize={12} />
                                <Tooltip
                                    formatter={(value) => [formatCurrency(value), 'Total']}
                                    contentStyle={{
                                        background: 'white',
                                        border: '1px solid #e5e7eb',
                                        borderRadius: '8px',
                                    }}
                                />
                                <Bar
                                    dataKey="total"
                                    fill="url(#colorGradient)"
                                    radius={[4, 4, 0, 0]}
                                />
                                <defs>
                                    <linearGradient id="colorGradient" x1="0" y1="0" x2="0" y2="1">
                                        <stop offset="0%" stopColor="#6366f1" />
                                        <stop offset="100%" stopColor="#ec4899" />
                                    </linearGradient>
                                </defs>
                            </BarChart>
                        </ResponsiveContainer>
                    </div>
                </div>
            </div>

            {/* Bottom Row */}
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(400px, 1fr))', gap: '1.5rem' }}>
                {/* Alerts */}
                <div className="card">
                    <div className="card-header">
                        <h3 className="card-title">Alertas</h3>
                    </div>
                    <div className="card-body">
                        {data?.alerts?.length > 0 ? (
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '0.75rem' }}>
                                {data.alerts.map((alert, index) => (
                                    <div
                                        key={index}
                                        className={`alert alert-${alert.type}`}
                                        style={{ marginBottom: 0 }}
                                    >
                                        <AlertCircle size={18} />
                                        <span>{alert.message}</span>
                                    </div>
                                ))}
                            </div>
                        ) : (
                            <div className="empty-state" style={{ padding: '2rem' }}>
                                <p>No hay alertas pendientes</p>
                            </div>
                        )}
                    </div>
                </div>

                {/* Recent Orders */}
                <div className="card">
                    <div className="card-header">
                        <h3 className="card-title">Órdenes Recientes</h3>
                    </div>
                    <div className="table-container">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Orden</th>
                                    <th>Evento</th>
                                    <th>Total</th>
                                    <th>Estado</th>
                                </tr>
                            </thead>
                            <tbody>
                                {(data?.recent_orders || mockData.recent_orders).slice(0, 5).map((order) => (
                                    <tr key={order.id || order.order_number}>
                                        <td style={{ fontWeight: 500 }}>{order.order_number}</td>
                                        <td>{order.event?.name || 'Evento'}</td>
                                        <td>{formatCurrency(order.total)}</td>
                                        <td>
                                            <span className={`badge badge-${order.status === 'paid' ? 'success' : 'warning'}`}>
                                                {order.status === 'paid' ? 'Pagado' : 'Pendiente'}
                                            </span>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    );
}

// Mock data for demo when API not available
const mockData = {
    sales: { today: 450000, week: 2500000, month: 8500000 },
    tickets: { sold_today: 45, sold_week: 280, used_today: 120 },
    counts: { total_clients: 1250, active_events: 5, orders_today: 38 },
    daily_sales: [
        { date: '2024-01-01', total: 380000 },
        { date: '2024-01-02', total: 420000 },
        { date: '2024-01-03', total: 350000 },
        { date: '2024-01-04', total: 480000 },
        { date: '2024-01-05', total: 520000 },
        { date: '2024-01-06', total: 680000 },
        { date: '2024-01-07', total: 450000 },
    ],
    sales_by_channel: [
        { channel: 'Web', total: 5800000, count: 420 },
        { channel: 'TPV', total: 2700000, count: 180 },
    ],
    alerts: [
        { type: 'warning', message: 'Evento "Noche VIP" casi lleno: 8 entradas disponibles' },
        { type: 'info', message: '3 órdenes pendientes hace más de 1 hora' },
    ],
    recent_orders: [
        { id: 1, order_number: 'ORD-20240107-A1B2', event: { name: 'Noche de Inauguración' }, total: 45000, status: 'paid' },
        { id: 2, order_number: 'ORD-20240107-C3D4', event: { name: 'After Office' }, total: 0, status: 'paid' },
        { id: 3, order_number: 'ORD-20240107-E5F6', event: { name: 'Noche VIP' }, total: 60000, status: 'pending' },
        { id: 4, order_number: 'ORD-20240107-G7H8', event: { name: 'Noche de Inauguración' }, total: 30000, status: 'paid' },
        { id: 5, order_number: 'ORD-20240107-I9J0', event: { name: 'Noche VIP' }, total: 120000, status: 'paid' },
    ],
};
