import { useState } from 'react';
import { Outlet, NavLink, useNavigate } from 'react-router-dom';
import { useAuth } from '../context/AuthContext';
import {
    LayoutDashboard,
    Calendar,
    Building2,
    Users,
    CreditCard,
    Settings,
    LogOut,
    Menu,
    X,
    Ticket,
    Bell,
} from 'lucide-react';

export default function Layout() {
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const { user, logout } = useAuth();
    const navigate = useNavigate();

    const handleLogout = async () => {
        await logout();
        navigate('/login');
    };

    const navItems = [
        { to: '/', icon: LayoutDashboard, label: 'Dashboard', exact: true },
        { to: '/events', icon: Calendar, label: 'Eventos' },
        { to: '/rooms', icon: Building2, label: 'Salones y Mesas' },
        { to: '/clients', icon: Users, label: 'Clientes' },
        { to: '/finance', icon: CreditCard, label: 'Finanzas' },
        { to: '/settings', icon: Settings, label: 'Configuración' },
    ];

    return (
        <div className="app-layout">
            {/* Mobile overlay */}
            {sidebarOpen && (
                <div
                    style={styles.overlay}
                    onClick={() => setSidebarOpen(false)}
                />
            )}

            {/* Sidebar */}
            <aside className={`sidebar ${sidebarOpen ? 'open' : ''}`}>
                <div className="sidebar-header">
                    <div className="sidebar-logo">
                        <div className="sidebar-logo-icon">
                            <Ticket size={20} />
                        </div>
                        <span>Sistema Reservas</span>
                    </div>
                </div>

                <nav className="sidebar-nav">
                    <div className="nav-section">
                        <span className="nav-section-title">Menú Principal</span>
                        {navItems.map((item) => (
                            <NavLink
                                key={item.to}
                                to={item.to}
                                end={item.exact}
                                className={({ isActive }) => `nav-item ${isActive ? 'active' : ''}`}
                                onClick={() => setSidebarOpen(false)}
                            >
                                <item.icon size={20} />
                                <span>{item.label}</span>
                            </NavLink>
                        ))}
                    </div>
                </nav>

                <div className="sidebar-footer">
                    <button
                        onClick={handleLogout}
                        className="nav-item"
                        style={{ width: '100%', border: 'none', background: 'transparent' }}
                    >
                        <LogOut size={20} />
                        <span>Cerrar Sesión</span>
                    </button>
                </div>
            </aside>

            {/* Main Content */}
            <main className="main-content">
                <header className="header">
                    <div style={{ display: 'flex', alignItems: 'center', gap: '1rem' }}>
                        <button
                            className="btn-icon"
                            onClick={() => setSidebarOpen(!sidebarOpen)}
                            style={styles.menuButton}
                        >
                            {sidebarOpen ? <X size={24} /> : <Menu size={24} />}
                        </button>
                        <h1 className="header-title">Panel de Administración</h1>
                    </div>

                    <div className="header-actions">
                        <button className="btn-icon" style={styles.notificationBtn}>
                            <Bell size={20} />
                            <span style={styles.notificationBadge}>3</span>
                        </button>

                        <div className="header-user">
                            <div className="header-avatar">
                                {user?.name?.charAt(0).toUpperCase() || 'A'}
                            </div>
                            <div style={{ display: 'none' }}>
                                <span style={{ fontWeight: 500, fontSize: '0.875rem' }}>{user?.name}</span>
                                <span style={{ fontSize: '0.75rem', color: '#6b7280' }}>{user?.role}</span>
                            </div>
                        </div>
                    </div>
                </header>

                <div className="page-content">
                    <Outlet />
                </div>
            </main>
        </div>
    );
}

const styles = {
    overlay: {
        position: 'fixed',
        inset: 0,
        background: 'rgba(0, 0, 0, 0.5)',
        zIndex: 99,
    },
    menuButton: {
        display: 'none',
        background: 'none',
        border: 'none',
        cursor: 'pointer',
        padding: '0.5rem',
        color: '#374151',
    },
    notificationBtn: {
        position: 'relative',
        background: 'none',
        border: 'none',
        cursor: 'pointer',
        padding: '0.5rem',
        color: '#6b7280',
    },
    notificationBadge: {
        position: 'absolute',
        top: '0',
        right: '0',
        background: '#ef4444',
        color: 'white',
        fontSize: '0.625rem',
        fontWeight: 600,
        width: '16px',
        height: '16px',
        borderRadius: '50%',
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
    },
};

// Add media query styles
const styleSheet = document.styleSheets[0];
styleSheet.insertRule(`
  @media (max-width: 1024px) {
    .menu-button { display: block !important; }
  }
`, styleSheet.cssRules.length);
